/* OdmFormat.java 0.10               UTF-8                   dh:2007-03-10
 *
 * The OdmFormat class provides static methods for verifying the format of
 * string parameters to ODMJNI 1.0 methods.
 *
 * These methods are intended for use by Java applications as a way of
 * verifying data obtained as input of some kind.  If an ill-formed string
 * is submitted to an ODMJNI method, an unchecked OdmError exception will
 * be thrown.  All invalid string parameters are treated as unrecoverable
 * application errors.  There is no point to catching and working around
 * such exceptions.  Instead, the application should be designed to assure
 * data validity by relying on the OdmFormat methods to filter input from
 * uncontrolled sources.
 *
 * OdmFormat classes are not intended to be instantiated.  All behavior
 * is delivered with static methods.
 *
 * XXX: ODMJNI 1.0 Java classes rely on the native Win32 OdmNative100
 *      library for implementation of all communication with ODMA and
 *      ODMA-compliant DMS integrations.  If the filters defined at this
 *      level are ever less-stringent than the OdmNative100 ones, an
 *      ODMJNI method may fail silently and inexplicably.  One approach
 *      to this would be to use JNI and common verifications from the
 *      OdmNative100 library.   It will be a challenging refactoring
 *      exercise to undertake that in a way that does not make for even
 *      more fragile coupling.
 *
 * FIXME: The discrepancy exists in 0.57beta and it will be repaired here
 *      and in odmjni100 and OdmNative100 as part of 0.58beta.  The problem
 *      exists in the current version of OdmFormat.wfDocId, which cannot
 *      match native approaches with a pure Java solution.  It is designed
 *      to be *too* permissive in the meantime.
 */

package info.odma.practical100;

public class OdmFormat extends OdmNullFormat

{   /* XXX: The use of inheritance is useful in the initial creation and
            confirmation of the use of OdmFormat within ODMJNI 1.0 Java
            classes.  The null implementation causes all of the format
            checks to fail and the throwing of OdmError can be observed
            and confirmed.  As the filters accept more input cases, already-
            successful confirmation tests should then run successfully
            again.  This idiosyncratic device is used here, where there's
            no harm done, to illustrate the principle.
       */

    private OdmFormat()
    {   /* Nothing to do.  Simply don't provide a constructor.  There
           are not meant to be any objects that extend this one.

           XXX: The only reason to construct one of these would be to
                be able to pass one as a parameter.  If that is needed,
                it can be done by containment and, ideally, delivery of
                an interface.
           */

        } /* OdmFormat */


    private static final int MAX_APPID_LENGTH = 15;
        /* The maximum length permitted for an Application ID String.  It
           is one less than the Odma32types100.h value of ODM_APPID_MAX.

           XXX: This value is private because it is not meaningful for
                OdmNullFormat and application programs do not require a
                dependency on on that level of detail.
           */


    public static boolean wfAppId(java.lang.String appId)

    {   /* Implement the same rules as those of verifyAppId in
           OdmNative100.cpp 0.28 (0.52beta).
           */

        if (appId == null) return false;

        if (appId.length() < 1 || appId.length() > MAX_APPID_LENGTH)
             return false;

        for (int i = 0; i < appId.length(); i++)
            {   char code = appId.charAt(i);
                if (  code < '0'|| code > 'z'
                        || ( code > 'Z' && code < 'a' )
                        || ( code > '9' && code < 'A' )
                      )
                     return false;
                }

        return true;

        } /* wfAppId */



    private static final int MAX_DOCFORMAT_LENGTH = 80;
        /* One less than the Odma32types100.h ODM_FORMAT_MAX value that
           also includes a terminal '\0' character.
           */


    public static

        boolean wfDocFormatName(java.lang.String docFormatName)

    {   /* Use the definition in section 1.2 of the 0.30alpha Pre-
           Release Notes.

           See <http://ODMA.info/dev/devNotes/2006/10/d061001f.htm>.
           */

        if (docFormatName == null) return false;

        if (  docFormatName.length() < 2
                || docFormatName.length() > MAX_DOCFORMAT_LENGTH )
             return false;

        if (docFormatName.charAt(0) != '.') return false;

        for (int i = 1; i < docFormatName.length(); i++)
            {   char code = docFormatName.charAt(i);
                if (  code < '0'|| code > 'z'
                        || ( code > 'Z' && code < 'a' )
                        || ( code > '9' && code < 'A' )
                      )
                     return false;
                } /* remaining characters same as for appId */

        return true;

        } /* wfDocFormatName */



    private static final int MAX_DOCID_LENGTH = 254;
        /* One less than the Odma32types.h ODM_DOCID_MAX value, which
           includes allowance for a terminal '\0' not used in the Java
           String form. */


    private static final int MAX_DMSID_LENGTH = 8;
        /* One less than the Odma32types.h ODM_DMSID_MAX value. */


    private static final java.lang.String ODMPREFIX = "::ODMA\\";
        /* The ODMA Document ID is case insensitive, although
           applications are not to change case of docId values.
           Remember that '\\' is a single \-character.
           */


    private static final int MAX_DOCIDPREFIX_LENGTH = 16;
    /* "::ODMA\\12345678\\".length() */


    public static boolean wfDocIdPrefix(java.lang.String docId)

    {   if (docId == null) return false;

        if (docId.length() < ODMPREFIX.length()+2) return false;
            /* No room for prefix when < "::ODMA\\?\\".length() */

        /* There's room, let's match the ODMPREFIX first. */
        if (  docId.substring(0, ODMPREFIX.length())
                .compareToIgnoreCase(ODMPREFIX)
              != 0)
             return false;

        /* The prefix is present, now let's see about the DMS ID */

        java.lang.String dmsId = "";

        int m = (docId.length() > MAX_DOCID_LENGTH
                    ? MAX_DOCID_LENGTH
                    : docId.length()
                    );

        for (int i = ODMPREFIX.length(); i < m; i++ )
            {   /* Accumulate valid characters until "\" or end. */

                char code = docId.charAt(i);

                if (code == '\\')
                     /* We're done:  */
                     return !( dmsId.length() < 1
                               || dmsId.length() > MAX_DMSID_LENGTH );

                /* Allow appId characters as DMS ID characters */
                if (  code < '0'|| code > 'z'
                        || ( code > 'Z' && code < 'a' )
                        || ( code > '9' && code < 'A' )
                      )
                     return false;

                dmsId = dmsId + code;
                }

        return false;
            /* Because we never found the prefix-ending '\\' */

        } /* wfDocIdPrefix */



    public static java.lang.String

        prefixDmsId(java.lang.String docId)

    {   /* Give a valid prefix, extract the dmsId portion */

        if (!wfDocIdPrefix(docId)) return null;

        java.lang.String dmsId = "";

        for (int i = ODMPREFIX.length(); ; i++ )
            {   /* Accumulate valid characters until '\\' */

                char code = docId.charAt(i);

                if (code == '\\') return dmsId;

                dmsId = dmsId + code;

                }

        } /* prefixDmsId */



    public static boolean wfDocId(java.lang.String docId)

    {   /* Technically, the only thing required for a valid ODMA Document
           ID is a valid prefix.

           Any finer restriction will be with a DMS determining that the
           format is not an acceptable form for it.

           FIXME: This format accepts all graphical characters beyond a
                  valid prefix so long as the length is acceptable.

                  A native version of this filter will be introduced in
                  0.58beta after openKnownDocument is implemented in
                  0.57beta.  The 0.57 openKnownDocument will filter and
                  silently fail for Document IDs that don't translate to
                  the default Windows ANSI Code Page.

                  The 0.58 openKnownDocument will throw an exception for
                  Document IDs that don't translate, and the 0.58beta
                  version of wfDocId will filter for exactly the same
                  permissible format.
           */

        if (docId.length() > MAX_DOCID_LENGTH) return false;

        if (!wfDocIdPrefix(docId)) return false;

        for (int i = 0; i < docId.length(); i++)

            {   char code = docId.charAt(i);

                if (code < 0x20
                       || (code > 0x7e && code < 0xa1)
                       || code == 0xad
                       )
                     return false;
                     /* XXX: Although the non-break space (0xa0) and
                             the soft hyphen (0xad) are technically not
                             control characters, their allowance in
                             Document IDs would be pernicious.
                        */
                }

        return true;

        } /* wfDocId */



    public static boolean safeDocId(java.lang.String docId)

    {   /* Ensure that the Document ID is actually safe for a large
           number of interchange applications by restricting as many
           characters as I can imagine.

           XXX: This is one of those features for which restrictions
                are easier to remove rather than increase, later.

           FIXME: I am nervous about the few non-alphanumeric
                characters that I allow even here.  If there are to be
                *fewer* this must be resolved before 0.60beta and opening
                of ODMJNI to widespread download and review.
           */

        if (!wfDocId(docId)) return false;

        for (int i = ODMPREFIX.length(); i < docId.length(); i++)

            {   /* rule out everything but Basic Latin alphanumberic
                   characters so that precautions can be taken in the
                   case of other character-code occurrences.

                   XXX: The idea is that this can always be relaxed,
                        but we will never have to tighten it further.
                   */

                char code = docId.charAt(i);

                switch (code)
                   {    case   ':':
                        case   '-':
                        case   '_':
                        case   '.':
                        case  '\\':
                        case   '/': continue;

                        default:
                            if (  code < '0'|| code > 'z'
                                    || ( code > 'Z' && code < 'a' )
                                    || ( code > '9' && code < 'A' )
                                  )
                                 return false;

                        }
                }

        return true;

        } /* safeDocId */


    } /* OdmFormat */



/* 0.10 2007-03-10-14:22 Correct bugs in wfDocIdPrefix use of .substring
   0.09 2007-03-07-11:06 Correct the version numbering and rebuild.  This
        is just a deployment cleanup for 0.56beta.
   0.08 2007-03-06-17:32 Adjust wfDocId to exclude C0 and C1 controls from
        all consideration as well.  The rest has to be left to the check for
        translateability to the default Windows ANSI code page.
   0.07 2007-03-06-16:55 Revise wfDocId to provide a relaxed test that does
        not catch all cases that will (silently) fail in openKnownDocument.
   0.06 2007-03-03-16:12 Tidy up wfDocId and safeDocId as much as I can
        and get on with it, with opportunity for later adjustment.
   0.05 2007-03-02-13:55 Extend checks in wfDocId and add safeDocId.
   0.04 2007-03-01-19:26 Add wfDocId and the related filters and constants.
   0.03 2007-03-01-19:11 Add wfDocFormatId and MAX_DOCFORMAT_LENGTH to use.
   0.02 2007-03-01-14:06 Add wfAppId and MAX_APPID_LENGTH to use.
   0.01 2007-02-28-21:37 Correct for clean compile.
   0.00 2007-02-25-14:24 Make initial trial version that simply
        inherits everything from OdmNullFormat.

   $Header: /ODMdev/info/odma/practical100/OdmFormat.java 11    07-03-10 18:12 Orcmid $
   */

/*         *** end of info.odma.practical100.OdmFormat.java ***          */