/* OdmNullConnection.java 0.13       UTF-8                   dh:2007-03-01
 *
 * The OdmNullConnection class implements null behavior only.  There is no
 * connection, no default DMS, and all operations provide their null
 * behavior.  Instantiations are generally used solely by their interfaces,
 * as via
 *
 *      OdmConnection myConnection = OdmNullBind.application("MyAppId");
 *
 * which does the construction.
 *
 * This class is useful for regression testing an application for correct
 * implementation of null connection situations.  The class is also a
 * foundation for extension to other implementations, such as OdmJniApp.
 *
 * If the application has provision for suppressing ODMA-aware behavior
 * as a configuration option, using this class allows all ODMA-aware code
 * of the application to remain unchanged.  The null connection will
 * automatically cause the correct non-ODMA behavior.
 *
 * This class has minimal resource requirements and there should be
 * no noticeable overhead when it is used to avoid ODMA-aware behavior
 * as an user's configuration choice.
 */


package info.odma.practical100;

public class OdmNullConnection implements OdmConnection
{
    /* Instance Variables and Constructor
     * ----------------------------------
     *
     * To avoid unnecessary construction of classes for null behaviors,
     * the OdmNullConnection class holds a cache of null behaviors that
     * other implementations can use for their error conditions.  The
     * overall-effect is that result codes delivered in other kinds of
     * programming models are essentially "constant" class instances
     * in the ODMJNI handling of failure modes.
     *
     * Because none of these null cases are instantiated until there is
     * a request for it, we also minimize the work of the constructor and
     * take some load off of application startup.
     *
     */


    protected final java.lang.String odmAppId;
        /* The validated Application ID string available to
           subclasses as needed. */

    protected final OdmNull nullDocument;
        /* Initialized in the constructor to hold the cache for null-
           document behaviors. */


    protected OdmNullConnection(java.lang.String appId)
                        throws OdmError

    {   if (!OdmFormat.wfAppId(appId))
             throw new OdmError
             ("Ill-formed appId parameter: "
                           + interfaceImplementation());

        nullDocument = new OdmNullCache();
            /* All that is needed here.  It is shared with non-null
               OdmConnection implementations and can also be shared with
               the document instances that they construct.
               */

        odmAppId = appId;

        } /* OdmNullConnection */


    protected OdmNullConnection( java.lang.String  appId,
                                  java.awt.Window  appWindow )
                        throws OdmError

    {   this(appId);

        if (appWindow == null)
            throw new OdmError
                        ("Ill-formed appWindow parameter: "
                           + interfaceImplementation() );



        }   /* OdmNullConnection */


    /* OdmInterface Interface Method Implementations
     * ---------------------------------------------
     */

    public java.lang.String interfaceImplementation()
    {   /* Return a text string describing this implementation
           XXX: This is used from constructor exceptions too. */

        return
            "ODMJNI 1.0 OdmNullConnection 0.13";

        }


    public boolean available()
    {   /* There is no connection available.
           */

        return false;
        }


    public void release()
    {   /* There are no resources that require release. The cache of
           null-behavior document interfaces is kept in operation until
           finalize handles it.  The null methods here are also unchanged
           after a release.
           */

        return;
        }


    /* OdmConnection Interface Method Implementations
     * ----------------------------------------------
     */

    public boolean dmsDefaultAvailable()
    {   /* There is no default DMS available. */

        return false;
        }


    public OdmPendingDocument acceptNewDocument(java.lang.String docFormat)
                              throws OdmError

    {   /* The operation must fail with a null pending document
           that specifies local operation */

        if (!OdmFormat.wfDocFormatName(docFormat))
             throw new
                  OdmError
                     ( "Ill-formed acceptNewDocument docFormat parameter: "
                          + interfaceImplementation() );

        return nullDocument.pendingLocalOperation();

        } /* acceptNewDocument */


    public OdmWorkingDocument chooseDocument()
    {   /* The operation must fail with a null working document
           that specifies local operation */

        return nullDocument.workingLocalOperation();
        }


    public OdmWorkingDocument openKnownDocument(java.lang.String docId)
                              throws OdmError

    {   /* This operation must also fail with a null working document,
           one that specifies failed operation. */

        if (!OdmFormat.wfDocId(docId))
             throw new OdmError
             ("Ill-formed openKnownDocument docId parameter: "
                           + interfaceImplementation() );

        return nullDocument.workingFailed();


        }


    } /* OdmNullConnection */


/* 0.13 2007-03-01-15:58 Make appId error message consistent with others.
   0.12 2007-03-01-15:26 Change error messages to use the implementation
        identification string and use the documented format.
   0.11 2007-02-28-21:43 Correct for clean compilation
   0.10 2007-02-28-21:27 Add second protected constructor and implement
        the format validations that apply to the constructors and to the
        interface methods.
   0.09 2006-12-18-20:48 Factor out the null-document cache implementation
        and use the extracted material to make the OdmNull interface and
        the OdmNullCache implementation.
   0.08 2006-12-17-20:11 corrections to get clean compile
   0.07 2006-12-17-17:36 Adjust the handing of failure modes and null-on-
        null behaviors to reflect the final state navigation model.
   0.06 2006-12-16-22:25 Correct a method capitalization and add caveat
        about how the null document instances must all surrender their
        reference to the OdmNullConnection instance in their constructors,
        which will not be attempted until the OdmNullConnection constructor
        has completed.
   0.05 2006-12-16-12:03 Complete updated null methods for all of the
        OdmConnection operations.
   0.04 2006-12-15-17:30 Transform to the form required for 0.30beta,
        with improved caching of the flavors of null documents that any
        connection and document implementations may want to rely on.
   0.03 2006-11-30-00:26 Make nullPending and nullWorking protected so that
        they can be used by subclasses of OdmNullConnection in odmjni100.
   0.02 2006-10-23-14:52 Remove the cyclic dependencies with OdmNullDocument
        instances, since we don't need it.
   0.01 2006-10-22-23:43 Make all methods public.
   0.00 2006-10-22-21:05 alpha-level version for the first evolutionary
        prototype of the OdmNullConnection implementation.  This class will
        be maintained consistent with the OdmConnection interface and its
        null behaviors as the prototype package is evolved.

   $Header: /ODMdev/info/odma/practical100/OdmNullConnection.java 14    07-03-01 15:59 Orcmid $
   */

/*    *** end of info.odma.practical100.OdmNullConnection.java ***     */