/* Setup01.java 0.06                  UTF-8                  dh:2006-11-14
 *
 *            DEMONSTRATE THE MASTER CASES OF JNI OPERATION
 *            *********************************************
 *
 * This program is a far derivative of examples by Shen Liang in Chapters 2
 * and 3 of "The Java Native Interface:  * Programmer's Guide and Specification."
 *
 * An important difference is the use of a private static native methods
 * for all of odmjni100.  In addition, none of the methods require any
 * use of the jclass object itself.
 *
 * The master cases for ODMJNI are all about converting between
 * java.lang.String objects and ASCII/UTF-8 C-style in-memory strings.
 * String objects are the only ones that pass between odmjni objects and
 * native method implementations.  All other parameters and results are
 * simple Java data types.  This makes for very simple native method
 * implementations.
 *
 * This module exercises all of the basic cases of transfer between Java
 * objects and odmjni100-like native methods.  The one key omission is
 * behavior involving access to the IodmNative interfaces to OdmNative
 * objects implemented in the JNI DLL.
 */

 class Setup01
 {  /* PROVIDE OPERATIONAL DEMONSTRATION OF NATIVE METHODS.
       */


    /* NATIVE METHODS DEMONSTRATING odmjni100 CASES
       ******************************************** */


    private static native int jniVersion();
        /* The version of JNI in two halves of an int, as
           delivered by JNI Ienv.GetVersion().
             Demonstrates:
               1. Delivery of an int result.
               2. Proper entry to an Ienv interface function.
           */


    private static native java.lang.String jniGreeting();
        /* A Java String that can be used to print the
           greeting information from the DLL.
             Demonstrates:
               1. Access to more interesting Ienv functions.
               2. Conversion from ASCII/UTF-8 string to a
                  Java string.
               3. Return of a string from a native method.
           */

    private static native long jniAppId(java.lang.String appId);
        /* The argument String is output to the console after
           conversion to ASCII/UTF8.  The address of the UTF8
           buffer is returned just to have a pointer to pass
           back.
             Demonstrates:
               1. Conversion of String value to dynamic block.
               2. Return of the dynamic block.
               3. Output of the text to the Windows Console.
               4. Return of a pointer via a jlong.
           */

    private static native java.lang.String jniNullString(long link);
        /* The link parameter is simply for exercising that input
           case.  This method always returns a null String object.
             Demonstrates:
               1. Internal acceptance of a pass-back point disguised
                  as a long (but not used as a reference).
               2. Confirmation of the methodology for returning a
                  null String/Object reference.
           */


    /* AUXILLIARY OUTPUT PRESENTATION OPERATIONS
       ***************************************** */


    private static void hello(IConOut console)
    {   /* Identify the application and what we are doing here.
           */

        console.println();
        console.println( jniGreeting() );
        }


    private static final String indent = "         ";
            /* Indent below the          "Setup01> ". */

    private static final String stars =  "       **** ";
            /* Used to highlight unexpected results */


    private static void showJniVersion(IConOut console,
                                           int jVersion)
    {   /* Take apart the version number and present it
               as text */

        int verMajor = jVersion >>> 16;

        int verMinor = jVersion - (verMajor << 16);

        console.print(indent + "Using JNI version ");
        console.print(verMajor);
        console.print(".");
        console.print(verMinor);
        console.println(" with Setup01.dll for implementations.");

        } /* showJniVersion */


    private static void showAppId(         IConOut console,
                                  java.lang.String appId )
    {   /* Build output line with the Application ID and
           also the value of a pointer passed through */
        console.print(indent + "Registering '");
        console.flush();
            /* To make sure cursor is in right place for the
               output form jniAppId. */
        long ptr = jniAppId(appId);
        console.print("' application.  Holding pointer 0x");
        console.println(java.lang.Long.toHexString(ptr));

        } /* showAppId */


    private static void showNullResult(         IConOut console,
                                       java.lang.String nullExpected )
    {   /* Simply report on the string that we got.  A null should
           report as having value "null".  */

        console.print(indent + "Expected null String has value = ");
        console.print(nullExpected);
        if (nullExpected == null)
             console.print(" as expected");
        console.println();

        } /* showNullResult */



    private static void goodbye(IConOut console)
    {   /* Announce completion of operation */

        console.println();
        console.println("Setup01> exit");
        console.println();
        }



    private static final IConOut
        stdout = new ConWriter(java.lang.System.out);
                     /* use a default console-output implementation */


    /* THE MAIN PROCEDURE AND ITS SUBORDINATE OPERATIONS
       ************************************************* */


    public static void main(String[] args)
    {   /* Exercise all of the Master Cases */

        hello(stdout);

        showJniVersion(stdout, jniVersion() );

        showAppId(stdout, "OdmNativeTest");

        showNullResult(stdout,
                       jniNullString(java.lang.Long.MAX_VALUE
                                       - java.lang.Integer.MAX_VALUE ) );

        goodbye(stdout);

        } /* main */

    static

        {   /* Static initialization of this class specifies the
               loading of the DLL that we'll use. */

            System.loadLibrary("Setup01");

            } /* static initialization */

    } /* Setup01 */


/* 0.06 2006-11-14-14:43 Add jniNullString to confirm how null String
        results are communicated.

   0.05 2006-11-13-23:49 Flush to console before any potential output from
        a native method.  This is needed to synchronize streams to the same
        console.

   0.04 2006-11-13-22:40 Add String Conversion the other way, introducing
        jniAppId, with a long returned value.

   0.03 2006-11-13-22:30 Clean up output presentation layout.

   0.02 2006-11-13-19:35 Add jniGreeting() to demonstrate receipt of a
        String result.

   0.01 2006-11-12-23:14 Demonstrate jniVersion() result.

   0.00 2006-11-12-22:23 Thinly derived from Setup00.java 0.01 to provide
        a master test of the key cases.  The output classes and approach
        are the same as used in the C applications and in practical100
        TestNullConnection.java test.  Confirm operation with no native
        methods

   $Header: /MyProjects/java/ODMdev/info/odma/odmjni100/test/setup01/Setup01.java 9     06-11-14 15:15 Orcmid $
   */

/*                         *** end of Setup01.java ***                   */
