/* CheckKnown.java 0.01             UTF-8                   dh:2007-03-10
 *
 *              DEMONSTRATE THE odmjni100 BASIC NATIVE FUNCTIONS
 *              ************************************************
 *
 * This program relies on info.odma.odmjni100.OdmJniBind methods for
 * confirmation of successful access to the odmjni100.dll.
 *
 * The basic exercise consists of selecting a document via chooseDocument
 * as a way to obtain known ODMA Document IDs.  Then openKnownDocument
 * is used to actually establish a working document.  .
  */

 import info.odma.practical100.*;

 import info.odma.odmjni100.*;

 class CheckKnown
 {  /* PROVIDE OPERATIONAL DEMONSTRATION OF ODMJNI NATIVE ACCESS.
       */


    /* AUXILLIARY OUTPUT PRESENTATION OPERATIONS
       ***************************************** */


    private static final java.lang.String indent = "            ";
            /* Indent below the                    "CheckKnown> ". */

    private static final java.lang.String stars =  "       **** ";
            /* Used to highlight unexpected results */


    private static void hello(IConOut console)
    {   /* Identify the application and what we are doing here.
           */

        console.println();
        console.println(   "CheckKnown> 0.01 Use Known ODMA Documents "
                           + " (via JNI " + OdmJniBind.getVersion()
                           + ")"
                           );

        } /* hello */



    private static void reportConnectionDrop(   IConOut  console,
                                                boolean  hasConnection
                                                )
    {   /* If there is a connection, report on its release */

        if (!hasConnection) return;

        console.println();
        console.println(indent
                        + "> > The Connection is no longer required.\n"
                        + indent
                        + "    It is released by OdmConnection.release()."
                        );

        } /* reportConnectionDrop */



    private static void reportRelease(IConOut console)
    {   /* Simply report that the symmetrical release case
           has happened.
           */

        console.println();
        console.println(indent + "> Interface OdmConnection.release() "
                               + "performed.");
        } /* reportRelease */



    private static void reportDefaultDropped
                            (  IConOut console,
                               boolean nowDefault
                               )
    {   /* Confirm that the default is no longer available. */

        if (!nowDefault)
             console.println(indent + "  The Default DMS is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE DEFAULT DMS DIDN'T DROP.");

        } /* reportDefaultDropped */



    private static void reportConDropped(  IConOut console,
                                           boolean nowConnected
                                           )
    {   /* Confirm that the connection is no longer available. */

        if (!nowConnected)
             console.println(indent + "  The connection is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE CONNECTION FAILED TO DROP.");

        } /* reportConDropped */



    private static void goodbye(IConOut console)
    {   /* Announce completion of operation */

        console.println();
        console.println("CheckKnown> exit");
        console.println();
        } /* goodbye */



    private static final IConLineIO stdio = new ConLineIO();
            /* Use a default console input-output implementation

            FIXME: We need to fix console operations to use Unicode
                   properly, if we can, and the Windows ANSI Code Page
                   if we can't.  */

    private static final IConOut stdout = stdio;

    private static final IConLineIn stdin = stdio;


    /* THE MAIN PROCEDURE AND ITS SUBORDINATE OPERATIONS
       ************************************************* */


    public static void main(String[] args)
    {   /* Work all the way through an iterative usage scenario where
           multiple documents are opened and manipulated (via Notepad). */

        hello(stdout);

        OdmConnection OdmApp = OdmJniBind.application("OdmNativeTest");

        CheckUtil.reportInterface(stdout, indent,
                                  OdmApp.interfaceImplementation() );

        boolean haveConnection = OdmApp.available();

        CheckUtil.reportConnection(stdout, indent, haveConnection);

        boolean haveDefaultDMS = OdmApp.dmsDefaultAvailable();

        if (!haveConnection)
             {  CheckUtil.reportExpectedNoDefault(stdout,
                                                  "    " + indent,
                                                  "    " + stars,
                                                  haveDefaultDMS
                                                  );
                haveDefaultDMS = false;
                    /* forcing consistency after possible problem. */
                }

        else CheckUtil.reportAvailabilityDefaultDMS(stdout,
                                                    indent + "      ",
                                                    indent + "> > > ",
                                                    haveDefaultDMS
                                                    );

        info.odma.practical100.OdmWorkingDocument choice = null;
            /* reference to result of chooseDocument(), if any.
               */

        final java.lang.String docIndent = indent + "        ";
        final java.lang.String docSlider = indent + "> > > > ";

        if (haveDefaultDMS)
             {  /* Perform chooseDocument() to get a known Document ID. */
                choice = OdmApp.chooseDocument();
                if (choice != null)
                     if (choice.operationSucceeded())
                          {   /* Use the docId to open as known document,
                                 if permissible at this point. */

                              java.lang.String chosen
                                    = choice.docID();
                              choice.release();

                              CheckUtil.reportStringOrNull
                                            (  stdout,
                                               docSlider,
                                               "known docID()",
                                               chosen
                                               );

                              stdout.println(docIndent
                                               + "wfDocIdPrefix() = "
                                               + OdmFormat.wfDocIdPrefix
                                                     (chosen)
                                               );

                              stdout.println(docIndent
                                               + "prefixDmsId() = "
                                               + OdmFormat.prefixDmsId
                                                     (chosen)
                                               );

                              stdout.println(docIndent
                                               + "wfDocId() = "
                                               + OdmFormat.wfDocId(chosen)
                                               );

                              stdout.println(docIndent
                                               + "safeDocId() = "
                                               + OdmFormat.safeDocId(chosen)
                                               );

                              choice = OdmApp.openKnownDocument(chosen);

                              }

                /* Report on the best working result we obtained. */
                CheckUtil.reportWorkingResult(stdout,
                                              docIndent,
                                              docSlider,
                                              choice
                                              );
                }

        boolean proceedOK = (choice != null);

        if (proceedOK)
             CheckUtil.cycleWorkingChanges(stdio, docIndent, choice);
                /* For a successful working document, allow the user
                   to view document and commit changes as often as
                   desired.
                   */

        while (proceedOK && choice.operationSucceeded())
            {   /* Invite Save As ... scenario and take each one through
                   to creation of a working document and change cycling,
                   until no more new documents are made.
                   */

                proceedOK
                    = CheckUtil.gotYes
                        (  stdio, docIndent
                                    + "Save to New DMS Document? [y]/N: "
                                  );

                OdmPendingDocument pend = null;

                if (proceedOK)
                     {  pend = choice.transferToNewDocument(".txt");
                        CheckUtil.reportPendingResult
                            (stdout, docIndent, docSlider, pend);
                        proceedOK = pend.operationSucceeded();
                        }

                if (proceedOK)
                     {  CheckUtil.doCommand("start Notepad "
                                    + pend.docSubmissionLocation()
                                    );
                        proceedOK
                            = CheckUtil.gotYes
                                (  stdio, docIndent
                                            + "Commit Content? [y]/N: "
                                          );
                        }

                OdmWorkingDocument work = null;

                if (proceedOK)
                     {  work = pend.commitContent();
                         /* FIXME: This is a good place to do another
                                   switch to a known document, using
                                   the same game as with the initial
                                   chooseDocument.  There is room for
                                   some common code here.
                                   */
                        CheckUtil.reportWorkingResult(stdout,
                                                      docIndent,
                                                      docSlider,
                                                      work
                                                      );
                        proceedOK = work.operationSucceeded();
                        }

                if (pend != null)
                     pend.release();
                        /* If we had a pending document, we must let it go.
                           */

                if (work != null)
                     {  /* If we have a new working document, we can
                           continue with it in place of the previous
                           choice.
                           */
                        choice.release();
                        choice = work;
                        work = null;
                        }

                if (proceedOK)
                     CheckUtil.cycleWorkingChanges
                            (stdio, docIndent, choice);
                            /* For a successful working document,
                               allow the user to view document and
                               commit changes as often as desired.
                               */
                }

        if (choice != null)
             choice.release();

        CheckUtil.reportDefaultDrop(stdout, indent, haveDefaultDMS);

        CheckUtil.reportConnectionDrop(stdout, indent, haveConnection);

        OdmApp.release();
        reportRelease(stdout);
        if (haveDefaultDMS)
             CheckUtil.reportDefaultDropped
                                 (  stdout, indent, stars,
                                    OdmApp.dmsDefaultAvailable()
                                    );
        if (haveConnection)
             CheckUtil.reportConDropped
                                 (  stdout, indent, stars,
                                    OdmApp.available()
                                    );

        goodbye(stdout);

        } /* main */



    } /* CheckChoice */


/* 0.01 2007-03-10-20:29 Incorporate exercise of wfDocIdPrefix and other
        functions as part of verification of the known Document ID.   This
        isolated the bug in OdmFormat.java 0.09.  Fix a bug that asked for
        Save As ... when unsuccessful Working Document creation.
   0.00 2007-03-10-13:02 Modify CheckChoice 0.17 to use chooseDocument
        as a way to obtain a Document ID that is then used with an
        openKnownDocument operation to continue with a working document.
        This should also work following a SaveAs, but we have not done
        that yet.

   $Header: /ODMdev/info/odma/odmjni100/test/Check04/CheckKnown.java 1     07-03-10 20:35 Orcmid $
   */

/*                       *** end of CheckKnown.java ***                */
