/* CheckUtil.java 0.04               UTF-8                   dh:2007-03-10
 *
 *            UTILITY FUNCTIONS FOR odmjni100 CONFIRMATION CHECKS
 *            ***************************************************
 *
 * This class provides utility functions used by the CheckChoice, CheckNew,
 * and any other confirmation checks.
 */

 import info.odma.practical100.*;

 import info.odma.odmjni100.*;

 class CheckUtil
 {  /* PROVIDE UTILITIES FOR EXERCISING THE ODMJNI INTEGRATION.
       */

    /* SYSTEM-LEVEL UTILITY FUNCTIONS
       ****************************** */

    static native int doCommand(java.lang.String command);
        /* Perform the command-line command and return the result code:

              0 the command-line processor returned 0 on completion

             >0 the command-line processor returned the non-zero result
                code from the command

             <0 the doCommand failed in some way.  These are the negative
                versions of possible Windows errno values:

                 -2  ENOENT, no such file or directory: no command processor

                 -7  E2BIG, command too long

                 -8  ENOEXEC, command processor file format error

                -12  ENOMEM, not enough memory for command processor or
                     other memory problem

           XXX: Since ODMA only runs on Windows, and ODMJNI requires
                Windows 2000 or later, we allow ourselves some direct native
                dependency in our testing and confirmation code.

           XXX: This call does not return until the command has completed.
                The use of start as a command is a way to initiate an
                application and return with the application still running.
                We do that when we launch notepad and then wait for the user
                to tell us whether or not to do something with the result.

           XXX: Remember to flush any console output before calling
                this function, especially if there is something the user
                must be able to refer to.  This function will flush the
                Windows streams, but the Java streams must be flushed by
                the application that uses doCommand.
          */



    /* AUXILLIARY OUTPUT PRESENTATION OPERATIONS
       ***************************************** */


    static void reportInterface(           IConOut  console,
                                   java.lang.String indent,
                                   java.lang.String impText
                                   )

    {   /* Report the interface implementation. */

        console.println();
        console.println(   indent + "> OdmConnection interface acquired:\n"
                         + indent + "  " + impText);

        } /* reportInterface */



    static void reportConnection(            IConOut console,
                                    java.lang.String indent,
                                             boolean hasConnection
                                    )

    {   /* Report whether or not there is a connection available. */

        console.println();

        if(hasConnection)
             {  /* Just report that there is one and we can continue. */
                console.println(indent + "> > "
                                       + "An ODMA Connection is available."
                                       );

                }
        else {  /* Point out that there is nothing more to do. */
                console.println(indent + "> > "
                                       + "No ODMA Connection is available."
                                       );

                console.println(indent + "    "
                                       + "No ODMA Operations can be "
                                       + "performed."
                                       );
                }

        } /* Report Connection */



    static void reportExpectedNoDefault
                            (  IConOut console,
                               java.lang.String indent,
                               java.lang.String stars,
                                        boolean hasDefaultDMS
                               )
    {   /* There should be no default because there is no connection. */

        if (hasDefaultDMS)
             console.println
                 (stars + "FALSE REPORT OF DEFAULT DMS AVAILABLE");
        else console.println(indent + "There is no default DMS, "
                                    + "as expected."
                                    );
        } /* reportExpectedNoDefault */


    static void reportAvailabilityDefaultDMS
                                (           IConOut console,
                                   java.lang.String indent,
                                   java.lang.String indentSlider,
                                            boolean hasDefaultDMS
                                   )
    {   /* There may or may not be a default DMS. */

        console.println();

        if (hasDefaultDMS)

             {  /* report that we have one and we just continue. */

                console.println(indentSlider
                                       + "A Default DMS is available."
                                       );
                }

        else {  /* report that we don't have one and we can't do much. */

                console.println(indentSlider
                                       + "No Default DMS is available."
                                       );

                console.println(indent + "No Default-DMS operations"
                                       + " are possible."
                                       );
                }

        } /* reportAvailabilityDefaultDMS */



    static void reportPendingResult
        (                                   IConOut  console,
                                   java.lang.String  indent,
                                   java.lang.String  indentSlider,
          info.odma.practical100.OdmPendingDocument  pend                                   )

    {   console.println();
        console.print(indentSlider + "returned interface = " );
        if (pend == null)
             {  console.println("null.");
                return;
                    /* because there is nothing more to say */
                }
        console.println("a pending-document implementation:");

        console.println(indent + pend );
        console.println(indent + pend.interfaceImplementation()
                           );

        console.println();

        console.println(indent + "           dmsAvailable() = "
                               + pend.dmsAvailable()
                               );
        console.println(indent + "     operationSucceeded() = "
                               + pend.operationSucceeded()
                               );
        console.println(indent + "     operationCancelled() = "
                               + pend.operationCancelled()
                               );
        console.println(indent + "localOperationRequested() = "
                               + pend.localOperationRequested()
                               );

        console.println();

        reportStringOrNull(console, indent,
                           "docSubmissionLocation()",
                           pend.docSubmissionLocation() );

        } /* reportPendingResult */



    static void reportWorkingResult
        (                                   IConOut  console,
                                   java.lang.String  indent,
                                   java.lang.String  indentSlider,
          info.odma.practical100.OdmWorkingDocument  choice                                   )

    {   /* Describe the document implemented by the interface.
        FIXME: There is a breach of cohesion and coupling here
               by performing operator interaction.  We should
               not be doing that here.  We may also have to
               use an IConIO interface to get it to work at all.
               */

        console.println();
        console.print(indentSlider + "returned interface = " );
        if (choice == null)
             {  console.println("null.");
                return;
                    /* because there is nothing more to say */
                }
                console.println("a working-document implementation:");

        console.println(indent + choice );
        console.println(indent + choice.interfaceImplementation()
                           );

        console.println();

        console.println(indent + "           dmsAvailable() = "
                               + choice.dmsAvailable()
                               );
        console.println(indent + "     operationSucceeded() = "
                               + choice.operationSucceeded()
                               );
        console.println(indent + "     operationCancelled() = "
                               + choice.operationCancelled()
                               );
        console.println(indent + "localOperationRequested() = "
                               + choice.localOperationRequested()
                               );

        console.println();

        showProperties(console, indent, choice);

        } /* reportWorkingResult */


    static void showProperties
               (                                     IConOut  console,
                                            java.lang.String  indent,
                   info.odma.practical100.OdmWorkingDocument  choice                                   )
    {
        console.println(indent
                           + "viewOnly() = " + choice.viewOnly()
                           );

        reportStringOrNull(console, indent,
                                    "docID()",
                                    choice.docID() );

        reportStringOrNull(console, indent,
                                    "docLocation()",
                                    choice.docLocation() );

        console.println();

        reportStringOrNull(console, indent,
                                    "windowTitle()",
                                    choice.windowTitle() );

        reportStringOrNull(console, indent,
                                    "dmsAuthor()",
                                    choice.dmsAuthor() );

        reportStringOrNull(console, indent,
                                    "dmsDocName()",
                                    choice.dmsDocName() );

        reportStringOrNull(console, indent,
                                    "dmsDocType()",
                                    choice.dmsDocType() );

        reportStringOrNull(console, indent,
                                    "dmsFormatName()",
                                    choice.dmsFormatName() );

        } /* showProperties */


    public static void reportStringOrNull
                (        IConOut  out,
                 java.lang.String indent,
                 java.lang.String name,
                 java.lang.String value)

    {   /* Show the string value as null or as the text value */

        out.print(indent + name + " = ");
        if (value == null)
             out.println("null");
        else out.println("\n" + indent + "  |"
                              + value + "|");

        } /* reportStringOrNull */


    static void cycleWorkingChanges(       IConLineIO con,
                                     java.lang.String indent,
                                     OdmWorkingDocument work
                                     )
    {   /* For an already-presented working document, work through
           commitChanges until no more are wanted.
           */

        boolean proceedOK = work.operationSucceeded()
                               && !work.viewOnly();

        while (proceedOK)
            { doCommand("start Notepad "
                               + work.docLocation()
                               );
              con.println();
              con.print(indent + "Commit Changes? ([y]/N): " );
              con.flush();
              java.lang.String resp = "N";
              try { resp = con.readLine();
                    }
              catch (java.io.IOException e)
                  { }
              con.println();
              if (resp == null)
                   break;
              if (resp.length() > 0)
                   if (resp.charAt(0) == 'N'
                         || resp.charAt(0) == 'n')
                        break;

              boolean changed = work.commitChanges();
              con.println(indent + "commitChanges reports "
                                 + changed
                                 );
              con.println();
              showProperties(con, indent, work);
              }

        } /* cycleWorkingChanges */



    static boolean gotYes(        IConLineIO con,
                            java.lang.String prompt )
    {   /* Prompt for a [y]N response. */

        con.println();
        con.print(prompt);
        con.flush();

        java.lang.String resp = null;

        try { resp = con.readLine();
                      }
        catch (java.io.IOException e)
            {         }

        con.println();
        if (resp == null)
             return false;

        if (resp.length() > 0)
             if (resp.charAt(0) == 'N'
                   || resp.charAt(0) == 'n')
                  return false;

        return true;

        } /* gotYes */



    static void reportDefaultDrop(          IConOut  console,
                                   java.lang.String  indent,
                                            boolean  hasDefaultDMS
                                   )
    {   /* If there is a default DMS, report on its release */

        if (!hasDefaultDMS) return;

        console.println();
        console.println(indent
                        + "> > > The Default DMS is no longer required.\n"
                        + indent
                        + "      It is released by OdmConnection.release()."
                        );

        } /* reportDefaultDrop */



    static void reportConnectionDrop(           IConOut  console,
                                       java.lang.String  indent,
                                                boolean  hasConnection
                                       )
    {   /* If there is a connection, report on its release */

        if (!hasConnection) return;

        console.println();
        console.println(indent
                        + "> > The Connection is no longer required.\n"
                        + indent
                        + "    It is released by OdmConnection.release()."
                        );

        } /* reportConnectionDrop */



    private static void reportRelease(         IConOut console,
                                      java.lang.String indent
                                      )
    {   /* Simply report that the symmetrical release case
           has happened.
           */

        console.println();
        console.println(indent + "> Interface OdmConnection.release() "
                               + "performed.");
        } /* reportRelease */



    static void reportDefaultDropped
                            (            IConOut console,
                                java.lang.String indent,
                                java.lang.String stars,
                                         boolean nowDefault
                                )
    {   /* Confirm that the default is no longer available. */

        if (!nowDefault)
             console.println(indent + "  The Default DMS is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE DEFAULT DMS DIDN'T DROP.");

        } /* reportDefaultDropped */



    static void reportConDropped(           IConOut console,
                                   java.lang.String indent,
                                   java.lang.String stars,
                                            boolean nowConnected
                                   )
    {   /* Confirm that the connection is no longer available. */

        if (!nowConnected)
             console.println(indent + "  The connection is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE CONNECTION FAILED TO DROP.");

        } /* reportConDropped */


    static

        {   /* Static initialization of this class specifies the
               loading of the only DLL that we'll use. */

            System.loadLibrary("CheckUtil");

            } /* static initialization */


    } /* CheckUtil */


/* 0.04 2007-03-10-13:46 Make reportStringOrNull public for use by
        CheckKnown in 0.57beta.
   0.03 2007-01-25-17:59 Add cycleWorkingChanges and anything else to
        simplify the cycling of commitChanges and transfers To NewDocuments.
   0.02 2007-01-24-22:34 Add reportPendingResult for use by CheckNew and
        CheckChoice later.
   0.01 2007-01-24-13:40 Add doCommand function with native implementation.
   0.00 2007-01-23-22:54 Strip reusable bits out of CheckChoice 0.14
        to make CheckChoice 0.15 and this utility class.

   $Header: /ODMdev/info/odma/odmjni100/test/Check04/CheckUtil.java 5     07-03-10 20:28 Orcmid $
   */

/*                       *** end of CheckUtil.java ***                */
