/* CheckChoice.java 0.17             UTF-8                   dh:2007-01-25
 *
 *              DEMONSTRATE THE odmjni100 BASIC NATIVE FUNCTIONS
 *              ************************************************
 *
 * This program relies on info.odma.odmjni100.OdmJniBind methods for
 * confirmation of successful access to the odmjni100.dll.  We only
 * obtain the results that come via OdmJniBind and the interfaces
 * that become available through OdmJniBind.application(appId).
 *
 * The basic exercise consists of selecting a document via chooseDocument
 * and then exercising the working document object that is returned.
  */

 import info.odma.practical100.*;

 import info.odma.odmjni100.*;

 class CheckChoice
 {  /* PROVIDE OPERATIONAL DEMONSTRATION OF ODMJNI NATIVE ACCESS.
       */


    /* AUXILLIARY OUTPUT PRESENTATION OPERATIONS
       ***************************************** */


    private static final java.lang.String indent = "             ";
            /* Indent below the          "CheckChoice> ". */

    private static final java.lang.String stars =  "        **** ";
            /* Used to highlight unexpected results */


    private static void hello(IConOut console)
    {   /* Identify the application and what we are doing here.
           */

        console.println();
        console.println(   "CheckChoice> 0.17 Use ODMA Working Documents "
                           + " (using JNI " + OdmJniBind.getVersion()
                           + ")"
                           );

        } /* hello */



    private static void reportConnectionDrop(   IConOut  console,
                                                boolean  hasConnection
                                                )
    {   /* If there is a connection, report on its release */

        if (!hasConnection) return;

        console.println();
        console.println(indent
                        + "> > The Connection is no longer required.\n"
                        + indent
                        + "    It is released by OdmConnection.release()."
                        );

        } /* reportConnectionDrop */



    private static void reportRelease(IConOut console)
    {   /* Simply report that the symmetrical release case
           has happened.
           */

        console.println();
        console.println(indent + "> Interface OdmConnection.release() "
                               + "performed.");
        } /* reportRelease */



    private static void reportDefaultDropped
                            (  IConOut console,
                               boolean nowDefault
                               )
    {   /* Confirm that the default is no longer available. */

        if (!nowDefault)
             console.println(indent + "  The Default DMS is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE DEFAULT DMS DIDN'T DROP.");

        } /* reportDefaultDropped */



    private static void reportConDropped(  IConOut console,
                                           boolean nowConnected
                                           )
    {   /* Confirm that the connection is no longer available. */

        if (!nowConnected)
             console.println(indent + "  The connection is no longer "
                                    + "available."
                                    );

        else console.println(stars + "THE CONNECTION FAILED TO DROP.");

        } /* reportConDropped */



    private static void goodbye(IConOut console)
    {   /* Announce completion of operation */

        console.println();
        console.println("CheckChoice> exit");
        console.println();
        } /* goodbye */



    private static final IConLineIO stdio = new ConLineIO();
            /* Use a default console input-output implementation */

    private static final IConOut stdout = stdio;

    private static final IConLineIn stdin = stdio;


    /* THE MAIN PROCEDURE AND ITS SUBORDINATE OPERATIONS
       ************************************************* */


    public static void main(String[] args)
    {   /* Exercise enough of OdmJniBind to confirm that we have
           basic connection and OdmNative access. */

        hello(stdout);

        OdmConnection OdmApp = OdmJniBind.application("OdmNativeTest");

        CheckUtil.reportInterface(stdout, indent,
                                  OdmApp.interfaceImplementation() );

        boolean haveConnection = OdmApp.available();

        CheckUtil.reportConnection(stdout, indent, haveConnection);

        boolean haveDefaultDMS = OdmApp.dmsDefaultAvailable();

        if (!haveConnection)
             {  CheckUtil.reportExpectedNoDefault(stdout,
                                                  "    " + indent,
                                                  "    " + stars,
                                                  haveDefaultDMS
                                                  );
                haveDefaultDMS = false;
                    /* forcing consistency after possible problem. */
                }

        else CheckUtil.reportAvailabilityDefaultDMS(stdout,
                                                    indent + "      ",
                                                    indent + "> > > ",
                                                    haveDefaultDMS
                                                    );

        info.odma.practical100.OdmWorkingDocument choice = null;
            /* reference to result of chooseDocument(), if any.
               */

        final java.lang.String docIndent = indent + "        ";
        final java.lang.String docSlider = indent + "> > > > ";

        if (haveDefaultDMS)
             {  /* Perform chooseDocument() and describe the result. */
                choice = OdmApp.chooseDocument();
                CheckUtil.reportWorkingResult(stdout,
                                              docIndent,
                                              docSlider,
                                              choice
                                              );
                }

        boolean proceedOK = (choice != null);

        if (proceedOK)
             CheckUtil.cycleWorkingChanges(stdio, docIndent, choice);
                /* For a successful working document, allow the user
                   to view document and commit changes as often as
                   desired.
                   */

        while (proceedOK)
            {   /* Invite Save As ... scenario and take each one through
                   to creation of a working document and change cycling,
                   until no more new documents are made.
                   */

                proceedOK
                    = CheckUtil.gotYes
                        (  stdio, docIndent
                                    + "Save to New DMS Document? [y]/N: "
                                  );

                OdmPendingDocument pend = null;

                if (proceedOK)
                     {  pend = choice.transferToNewDocument(".txt");
                        CheckUtil.reportPendingResult
                            (stdout, docIndent, docSlider, pend);
                        proceedOK = pend.operationSucceeded();
                        }

                if (proceedOK)
                     {  CheckUtil.doCommand("start Notepad "
                                    + pend.docSubmissionLocation()
                                    );
                        proceedOK
                            = CheckUtil.gotYes
                                (  stdio, docIndent
                                            + "Commit Content? [y]/N: "
                                          );
                        }

                OdmWorkingDocument work = null;

                if (proceedOK)
                     {  work = pend.commitContent();
                        CheckUtil.reportWorkingResult(stdout,
                                                      docIndent,
                                                      docSlider,
                                                      work
                                                      );
                        proceedOK = work.operationSucceeded();
                        }

                if (pend != null)
                     pend.release();
                        /* If we had a pending document, we must let it go.
                           */

                if (work != null)
                     {  /* If we have a new working document, we can
                           continue with it in place of the previous
                           choice.
                           */
                        choice.release();
                        choice = work;
                        work = null;
                        }

                if (proceedOK)
                     CheckUtil.cycleWorkingChanges
                            (stdio, docIndent, choice);
                            /* For a successful working document,
                               allow the user to view document and
                               commit changes as often as desired.
                               */
                }

        if (choice != null)
             choice.release();

        CheckUtil.reportDefaultDrop(stdout, indent, haveDefaultDMS);

        CheckUtil.reportConnectionDrop(stdout, indent, haveConnection);

        OdmApp.release();
        reportRelease(stdout);
        if (haveDefaultDMS)
             CheckUtil.reportDefaultDropped
                                 (  stdout, indent, stars,
                                    OdmApp.dmsDefaultAvailable()
                                    );
        if (haveConnection)
             CheckUtil.reportConDropped
                                 (  stdout, indent, stars,
                                    OdmApp.available()
                                    );

        goodbye(stdout);

        } /* main */



    } /* CheckChoice */


/* 0.17 2007-01-25-17:27 Add Save As ... Scenario that produces similar
        sequence to initial working document operation.
   0.16 2007-01-24-17:13 Use stdio packaging, launch working document and
        prompt for commitChanges.
   0.15 2007-01-23-18:36 Factor out CheckUtil.java for easier adaptability
        to additional scenarios.
   0.14 2007-01-23-18:27 Adapt as Check03 as CheckChoice and run regression
        check before further changes.
   0.13 2007-01-13-21:51 Add further logic to exercise commitChanges() in
        the manner of OdmNative test Check05 0.08 for 0.40alpha.
   0.12 2007-01-08-24:00 Be sure to release the document because finalize
        is apparently too late.
   0.11 2007-01-04-21:15 Expand the handling of chooseDocument results:
        Report on the returned interface implementation and document
        properties.
   0.10 2006-12-18-15:08 Make corrections to get clean compile.
   0.09 2006-12-18-15:01 Rename Setup02 to Check03 and customize to use the
        0.30alpha practical100 and 0.25alpha odmjni100.OdmJniBind.
   0.08 2006-11-30-13:22 Introduce provisional use of chooseDocument.
   0.07 2006-11-29-22:17 Add checks and reports on Default DMS availability.
   0.06 2006-11-29-21:13 Correct the sense of the expected boolean
        in reportConDropped.
   0.05 2006-11-29-19:20 Regression build to make sure adding lifecycle
        control of the rIodmNative interface reference in OdmJniApp has
        not changed anything.
   0.04 2006-11-29-17:02 Add report about Connection drop succeeding.
   0.03 2006-11-28-23:41 Report Connection status and dropping too.
   0.02 2006-11-28-22:33 Test the info.odma.odmjni100.OdmJniBind class
        static application("appId") method and demonstrate that it
        returns a null implementation of practical100.OdmConnection.
   0.01 2006-11-28-18:47 Use the public OdmJniBind.getVersion() method
        so we can see use of the native jniVersion method in an useful
        way.
   0.00 2006-11-28-18:04 Strip back Setup01.java 0.06 for building up
        as a Java-based exercise of the basic connection functions of
        odmjni100.

   $Header: /ODMdev/info/odma/odmjni100/test/Check04/CheckChoice.java 19    07-01-25 19:56 Orcmid $
   */

/*                       *** end of CheckChoice.java ***                */
