/* Check05.cpp 0.07                  UTF-8                    dh:2007-01-02
 *
 *            VERIFY THE IodmNative COM INTERFACE IMPLEMENTATION
 *            **************************************************
 *
 * Check05 provides the a regression confirmation of the 0.30alpha OdmNative
 * integration.   It exercises the basic cases of the 0.30alpha
 * IodmApplication interface.
 *
 * The capabilities confirmed are:
 *
 *   1. All of the functions of the Setup04 test program used to confirm
 *      the 0.20alpha build.
 *
 *   2. Additional 0.30alpha functions as this program is customized to
 *      verify additions as they are made.
 *
 *
 * This is a C++ program.  The COM interfaces that we use are defined as
 * C++ classes and the headers do not provide non-class substitutions when
 * C is being used instead of C++.
 *
 * For the most part, we are still writing "Clean C," but with C++ classes
 * as interface definitions.
 */

#include <stdio.h>
    /* For FILE, fputs, stdout, ... */

#include <string.h>
    /* for strncmp in verification of docID and docLocation results */

#include <windows.h>
    /* For standard Windows types, constants, and classes. */

#include "Odma32types100.h"
    /* ODMA types used with OdmNative100 interfaces */

#include "OdmNative100.hpp"
    /* For the basic IodmApplication100 interface and the factory function
       for starting OdmNative operation: OdmBindNative100.
       */

#include "OdmWorking100.hpp"
    /* For the basic IodmWorking interface that is used for working
       documents instantiated via OdmNative operations.
       */

    /* The OdmNative100.lib must also be included in the linking of this
       program.  See the BuildCheck05.bat file for location of the material
       needed at compile-time.
       */


/* AUXILLIARY INPUT-OUTPUT FUNCTIONS
 * *********************************
 */

/*              Check05> */
#define INDENT "         "
#define STARS  "    **** "

static const char indent[] = INDENT;
static const char stars[] =  STARS;



static void hello(FILE *out)
{   /* Tell them who's here. */

    fputs("\nCheck05> 0.07 Demonstration of OdmNative100 "
               "0.30alpha Functions.\n",
          out);

    fputs(INDENT "Program " __FILE__ " dated " __TIMESTAMP__ "\n"
          INDENT "Compiled at " __TIME__ " on " __DATE__
    #if defined(__cplusplus)
        " as C++"
    #elif defined(__STDC__)
        " as Standard C"
    #elif defined(_MSC_VER)
        " as Visual C"
    #else
        " as C Language"
    #endif
    #if defined(_MSC_EXTENSIONS)
        " extended"
    #endif
    #if defined(_DLL)
        " DLL"
    #endif
    #if defined(_WIN32)
        " (Win32)"
    #endif
        "\n\n", out);

    } /* hello */



static const char* hResultName(HRESULT rc)
{   /* Return the names of the known HRESULT values */

    switch (rc)
    {   case S_OK:
            return "S_OK";
        case E_NOINTERFACE:
            return "E_NOINTERFACE";
        case E_INVALIDARG:
            return "E_INVALIDARG";
        case E_POINTER:
            return "E_POINTER";
        case E_FAIL:
            return "E_FAIL";
        case E_OUTOFMEMORY:
            return "E_OUTOFMEMORY";
        case E_UNEXPECTED:
            return "E_UNEXPECTED";
        case E_NOTIMPL:
            return "E_NOTIMPL";
        case E_HANDLE:
            return "E_HANDLE";
        }

    return "unknown";

    } /* hResultName */



    static const char* odmStatusName(ODMSTATUS rc)
{   /* Return the names of the known ODMSTATUS values */

    switch (rc)
    {   case ODM_SUCCESS:
            return "ODM_SUCCESS";
        case ODM_E_FAIL:
            return "ODM_E_FAIL";
        case ODM_E_CANCEL:
            return "ODM_E_CANCEL";
        case ODM_E_NODMS:
            return "ODM_E_NODMS";
        case ODM_E_CANTINIT:
            return "ODM_E_CANTINIT";
        case ODM_E_VERSION:
            return "ODM_E_VERSION";
        case ODM_E_APPSELECT:
            return "ODM_E_APPSELECT";
        case ODM_E_USERINT:
            return "ODME_E_USERINT";
        case ODM_E_HANDLE:
            return "ODM_E_HANDLE";
        case ODM_E_ACCESS:
            return "ODM_E_ACCESS";
        case ODM_E_INUSE:
            return "ODM_E_INUSE";
        case ODM_E_DOCID:
            return "ODM_E_DOCID";
        case ODM_E_OPENMODE:
            return "ODM_E_OPENMODE";
        case ODM_E_NOOPEN:
            return "ODM_E_NOOPEN";
        case ODM_E_ITEM:
            return "ODM_E_ITEM";
        case ODM_E_OTHERAPP:
            return "ODM_E_OTHERAPP";
            /* Symbols below here are not defined in Odma32types100.h */
        case 16:
            return "ODM_E_NOMOREDATA (ODMA 1.5 feature)";
        case 17:
            return "ODM_E_PARTIALSUCCESS (ODMA 1.5 feature)";
        case 18:
            return "ODM_E_REQARG (ODMA 2.0 feature)";
        case 19:
            return "ODM_E_NOSUPPORT (ODMA 2.0 feature)";
        case 20:
            return "ODM_E_TRUNCATED (ODMA 2.0 feature)";
        case 21:
            return "ODM_E_INVARG (ODMA 2.0 feature)";
        case 22:
            return "ODM_E_OFFLINE (ODMA 2.0 feature)";
        case 23:
            return "ODM_E_ARCHIVED (ODMA 2.0 UNSPECIFIED)";
        case 24:
            return "ODM_E_ALREADYOPENED (ODMA 2.0 UNSPECIFIED)";
        case 25:
            return "ODM_E_FILELOCKED (ODMA 2.0 UNSPECIFIED)";
        case 26:
            return "ODM_E_REFUSED (ODMA 2.0 UNSPECIFIED)";
        case 27:
            return "ODM_W_NOACTION (ODMA 2.0 UNSPECIFIED)";
        case 28:
            return "ODM_E_NORELATION (ODMA 2.0 UNSPECIFIED)";
        }

    return "(not defined)";

    } /* odmStatusName */



static void reportIface(        FILE *out,
                          const char indent[],
                          const char ifaceName[],
                             HRESULT rc,
                                void *pIface
                          )
{   /* Just show the Interface Results that we get */

    fprintf(out, "%s%08p  p%s, ",
                 indent, pIface, ifaceName );

    fprintf(out, "rc = %08X (%s)\n",
                 rc, hResultName(rc) );

    } /* reportIface */



static void reportIodmApp( FILE  *out,
                           BOOL  haveIodmApp
                           )
{   if (!haveIodmApp)
         {  /* Lament having no IodmApplication100 interface */
            fputs("\n" STARS "IodmApplication100 INTERFACE NOT ACQUIRED\n"
                       STARS "    No further operations are possible.\n",
                  out);
            }

    else {  /* Report operations able to begin. */
            fputs("\n" INDENT "> IodmApplication100 "
                              "interface acquired.\n",
                  out);
            }

    } /* reportIodmApp */



static void reportIfaceVersion(   FILE  *out,
                                  LPCSTR  indent,
                                  LPCSTR  pszText
                                  )
{   /* Present the version text nicely below the interface-acquired report.
       */

    if (pszText == NULL) return;

    fputs(indent, out);

    fputs(pszText, out);
        /* FIXME: The lines need to be broken up and indented.
           ***************************************************
           This can be a bonus activity when cleaning up after
           beta release.
           */

    fputs("\n", out);

    } /* reportIodmAppVersion */



static void reportHasConMan( FILE  *out,
                       BOOL  haveConnectionManager
                           )
{   if (!haveConnectionManager)
         {  /* Lament having no Connection Manager */
            fputs("\n" STARS "CONNECTION MANAGER NOT AVAILABLE\n"
                       STARS "    No ODMA operations are possible and the\n"
                       STARS "    application must behave "
                                 "conventionally.\n",
                  out);
            }

    else {  /* Report operations able to begin. */
            fputs("\n" INDENT "> > An ODMA Connection Manager "
                              "is available to use.\n",
                  out);
            }

    } /* reportHasConMan */



static void reportUnexpectedDMS( FILE  *out,
                                 BOOL  unexpectedDMS
                                 )
{   if (unexpectedDMS)
         {  /* There's a problem here. */
            fputs("\n" STARS "DMS INCORRECTLY REPORTED AVAILABLE.\n"
                       STARS "    No DMS can be available if there is no\n"
                       STARS "    ODMA Connection Manager available.\n",
                  out );
            }

    else {  /* It is expected that there be no unexpected DMS ... */
            fputs(STARS "    As required in this case, there is also\n"
                  STARS "    no DefaultDMS.\n",
                  out );
            }

    } /* reportUnexpectedDMS */



static void reportHasDefaultDMS( FILE  *out,
                                 BOOL  hasDefaultDMS
                                 )
{   fputs("\n", out);

    if (!hasDefaultDMS)
         {  /* There's a limitation here. */
            fputs(INDENT "> > > NO DEFAULT DMS IS AVAILABLE.\n"
                  INDENT "      Operations that apply against a default\n"
                  INDENT "      are unavailable to this application.\n",
                  out);
            }

    else {  /* all ODMA 1.0 operations are available */
            fputs(INDENT "> > > There is a Default DMS for use.\n"
                  INDENT "      All ODMA 1.0 DMS-requiring operations\n"
                  INDENT "      are available to this application.\n",
                  out);
            }

    } /* reportDefaultDMS */



static void reportNoSelect(FILE* out, ODMSTATUS rc)
{   /* Show results of selectDocID operation that didn't return a
       Document ID.
       */

       fputs("\n" INDENT "> > > > No DocID returned from selectDocID.\n"
                  INDENT "        rc = ",
             out);

       fprintf(out, "%d, %s",
                    (int) rc, odmStatusName(rc) );

       switch (rc)
       {    /* Indicate if response is unexpected */
           case ODM_SUCCESS:
           case ODM_E_FAIL:
           case ODM_E_CANCEL:
           case ODM_E_NODMS:
           case ODM_E_APPSELECT:
                fputs("\n", out);
                return;
           }

       fputs(" *** INVALID ***\n", out);

       } /* reportNoSelect */


static void reportNoOpen(FILE* out, ODMSTATUS rc)
{   /* Show results of openDoc operation that didn't return a
       Document Location.
       */

       fputs("\n" INDENT "> > > > No DocID returned from selectDocID.\n"
                  INDENT "        rc = ",
             out);

       fprintf(out, "%d, %s",
                    (int) rc, odmStatusName(rc) );

       switch (rc)
       {    /* Indicate if response is unexpected */
           case ODM_SUCCESS:
           case ODM_E_FAIL:
           case ODM_E_ACCESS:
           case ODM_E_INUSE:
           case ODM_E_DOCID:
                fputs("\n", out);
                return;
           }

       fputs(" *** INVALID ***\n", out);

       } /* reportNoOpen */


static void reportDocProp(          FILE  *out,
                                  LPCSTR  indent,
                          IodmWorking100  *pIodmWrk,
                                  LPCSTR  propName,
                                    WORD  propNum
                                  )

{   /* Report the specified Document Property value, if available. */

    char propVal[300] = {'\0'};
        /* Choose a respectably-large value that will hold whatever we
           can expect to get */

    fputs(indent, out);
    fputs(propName, out);

    ODMSTATUS odmResult
        = pIodmWrk -> docProperty(propNum, propVal, sizeof(propVal) );

    if (propVal[0] == '\0' || odmResult != ODM_SUCCESS)
         {  /* For any null result, say so and translate the result code.
               */

            fputs(" is null (", out);
            fputs(odmStatusName(odmResult), out);
            fputs("\n", out);

            }

    else {  /* Show the value as well as we are able */

            fputs(" = |", out);
            fputs(propVal, out);
            fputs("|\n", out);

                }

    } /* reportDocProp */


static void reportGoesIodmWrk(FILE * out)
{   /* Explain what happens when IodmWorking100 is released */

    fputs("\n" INDENT "> > > > No further use of the document interface\n"
               INDENT "        will be made.  The resources will be\n"
               INDENT "        released, and any document closed, when\n"
               INDENT "        the last reference to the interface is\n"
               INDENT "        released.\n",
          out);

    } /* reportGoesIodmWrk */


static void reportGoesDMS(FILE* out)
{   /* Explain how the DMS is released. */

    fputs("\n" INDENT "> > > There is no further need for the Default\n"
               INDENT "      DMS.  It will be released automatically\n"
               INDENT "      when all IodmApplication100 references\n"
               INDENT "      are released.\n",
          out);

    } /* reportGoesDMS */



static void reportGoesConMan(FILE* out)
{   /* Explain how the Connection Manager is released. */

    fputs("\n" INDENT "> > There is no further need for the ODMA\n"
               INDENT "    Connection Manager.  It will be released\n"
               INDENT "    automatically when all IodmApplication100\n"
               INDENT "    interface references are released.\n",
          out );

    } /* reportGoesConMan */



static void reportRelease(FILE *out, const char ifaceName[])
{   /* Just build a bit more statement. */

    fputs(INDENT "> ", out);
    fputs(ifaceName, out);
    fputs(" interface released.\n", out);

    }


static void goodbye(FILE *out)
{   /* Tell them who's leaving. */

    fputs("\nCheck05> exit\n\n", out);
    }



/* APPLICATION-IMPORTANT CONSTANTS
 * *******************************
 */

static const char pszTestAppId[ ] = "OdmNativeTest\0";
    /* The name of the application for all of our tests. Guard with
       an extra null byte.  This is a valid Application ID.
       */

static const IID IID_IodmUnknown = IID_IodmUnknown_ ;
    /* Standard IUnknown IID uniquelly named to avoid conflicts
       with definitions in other headers and external data. */

static const IID IID_IodmApplication100 = IID_IodmApplication100_ ;
    /* The provisional IodmApplication100 IID of the current release. */

static const IID IID_IodmWorking100 = IID_IodmWorking100_ ;
    /* The provisional IodmWorking100 IID of the current release level. */

static const IID IID_IFoo = /* A never assigned IID used for fun. */
        { 0x12751e61, 0x9ef6, 0x451e,
            { 0x9a, 0xbd, 0xf9, 0x75, 0xad, 0x1d, 0x79, 0x91 } };



/* MAIN CHECK05 FUNCTION DEFINITION
 * ********************************
 */

 int main( int argc, char *argv[])

 {  /* Use simple console application with no command-line parameters.
       */

    /* GUARDED REFERENCE POINTERS AND STATE.  This data is managed
       is as part of an always stable state for the application.
       Values reflect the availability conditions of ODMA to the
       application.
       */

    IUnknown *pIFooUnknown = NULL;
        /* Not expected to produce a valid result.
           This is used solely as a negative confirmation.
           */

    IUnknown *pIodmAppUnk = NULL;
        /* The IUnknown interface of the OdmApplication Object, it
           would seem.  If we acquire this, we use it to Query Interface
           for the next one.
           */

    IodmApplication100 *pIodmApp = NULL;
        /* The Interface that allows us to do a little work.  This is
           the helper level for odmjni100 implementation of the
           info.odma.practical100.OdmConnection interface.
           */

    IodmWorking100 *pIodmWrk = NULL;
        /* The interface to the working document, if we manage to get
           that far. */

    HRESULT rc = S_OK;
        /* Result code for HRESULT-producing functions and methods. */

    BOOL proceedOK = FALSE;
        /* Boolean result for BOOL-returning interface methods. */

    int releaseCount = 0;
        /* Count of releases done at the end. */

    /* IDENTIFY APPLICATION.  This just shows which test is running.
       */

    hello(stdout);


    /* ACQUIRE THE APPLICATION-LEVEL INTERFACE.  Demonstrate that
       we are (or are not) obtaining the necessary interfaces.
       */

    rc = OdmBindNative100(pszTestAppId, IID_IFoo,
                                        (void**) &pIFooUnknown);
    reportIface(stdout, INDENT, "IFoo", rc, pIFooUnknown);

    rc = OdmBindNative100(pszTestAppId, IID_IodmUnknown,
                                       (void**) &pIodmAppUnk);
    reportIface(stdout, INDENT, "IUnknown", rc, pIodmAppUnk);

    if (SUCCEEDED(rc))
         rc = pIodmAppUnk -> QueryInterface( IID_IodmApplication100,
                                             (void**) &pIodmApp );

    else rc = OdmBindNative100(pszTestAppId, IID_IodmApplication100,
                                             (void**) &pIodmApp );

    reportIface(stdout, INDENT, "IodmApplication100", rc, pIodmApp);


    /* REPORT WHETHER HAVE INTERFACE AND CAN PROCEED
    */

    proceedOK = (pIodmApp != NULL);

    reportIodmApp(stdout, proceedOK);
        /* Reporting whether it is possible to proceed.
           TESTME: There are more cases of proper QueryInterface
                   and reference counting.  We need a test fixture for
                   that, and also on the defensive measures in the
                   IUnknown implementations.
           */

    if (proceedOK)
         reportIfaceVersion(  stdout, INDENT "  ",
                              pIodmApp -> interfaceImplementation()
                              );

         /* FIXME: This information needs to be presented in a way
                   that fits the indentation that we are using.
                   */


    /* DETERMINE AVAILABILITY OF CONNECTION MANAGER
       */

    if (proceedOK)
         {  /* Determine whether we have the Connection Manager
               and can continue.
               */

            proceedOK = ( pIodmApp -> hasConMan() );

            reportHasConMan(stdout, proceedOK);

            if (!proceedOK)
                reportUnexpectedDMS(stdout, pIodmApp -> hasDefaultDMS() );

            }

    /* DETERMINE AVAILABILITY OF A DEFAULT DMS FOR THE APPLICATION
       */

    if (proceedOK)
         {  /* Determine if we have a default DMS so we can continue
               with operations.
               */

            proceedOK = ( pIodmApp -> hasDefaultDMS() );

            reportHasDefaultDMS(stdout, proceedOK);

            }


    /* HAVE THE DEFAULT DMS CHOOSE A DOCUMENT ID IN COOPERATION WITH
       THE USER.

       FIXME: When OdmNative is reabstracted, all of this code should
              reduce to a single call that gets an HRESULT to report
              and a Document Interface when successful.  That will
              go much smoother than this version.
       */

    char theDocId[ODM_DOCID_MAX+2] = {'\0'};
            /* Buffer for any returned Document ID, with a little
               extra padding.
               */

    BOOL viewMode = FALSE;
            /* Set to indicate whether the document should be opened in
               view mode rather than modify mode. */

    ODMSTATUS odmResult = ODM_SUCCESS;

    if (proceedOK)
         {  /* Use the Default DMS to perform a selectDoc and start
               using the result.
               */


            odmResult = ( pIodmApp -> selectDocID(theDocId, &viewMode) );
            /* OK, let's see what we have. */

            if (odmResult != ODM_SUCCESS)
                 reportNoSelect(stdout, odmResult);
                    /* We have no Document ID so there is no further
                       we can go. */

            proceedOK = (odmResult == ODM_SUCCESS);

            }

    if (proceedOK)

         {  fputs("\n"  INDENT "> > > > A Document ID is selected for ",
                  stdout);

            if (viewMode)
                 fputs("viewing", stdout);
            else fputs("modification", stdout);

            fputs(".\n" INDENT "        Doc ID ", stdout);

            fputs(theDocId, stdout);

            fputs("\n", stdout);

            }

    char theDocLoc[ODM_FILENAME_MAX+20] = {'\0'};
        /* Where we store our version of the Document Location */

    if (proceedOK)

         {  odmResult = pIodmApp -> openDoc(theDocId, theDocLoc, viewMode);
                /* The ID was from a select, so open should be possible */

            if (odmResult != ODM_SUCCESS)
                     reportNoOpen(stdout, odmResult);
                        /* And if not, say why. */

            proceedOK = (odmResult == ODM_SUCCESS);

            }

    if (proceedOK)

         {  /* Report the Document Location for the opened Document */

            fputs("\n"  INDENT "> > > > File-system location of content:\n"
                        INDENT "        ",
                  stdout);

            fputs(theDocLoc, stdout);

            fputs("\n", stdout);

            }


    if (proceedOK)

         {  /* Make a Working Document that we can then use to perform
               all further operations on the specific document and its
               delivered content.
               */

               rc = pIodmApp -> makeWorkingDocument
                                    (  theDocId, theDocLoc, viewMode,
                                       IID_IodmWorking100,
                                       (void **) &pIodmWrk
                                       );

               reportIface(  stdout, "\n" INDENT "> > > > ",
                             "IodmWorking100", rc, pIodmWrk
                             );

               proceedOK = SUCCEEDED(rc);

               }

    if (pIodmWrk != NULL)
         reportIfaceVersion(  stdout, INDENT "        ",
                              pIodmWrk -> interfaceImplementation()
                              );

#   define docINDENT INDENT "        "

    if (pIodmWrk != NULL)
         {  /* Report the viewOnly setting */

            fputs("\n" docINDENT "viewOnly = ", stdout);
            fputs( (pIodmWrk -> viewOnly() ? "TRUE" : "FALSE"), stdout);
            fputs("\n", stdout);

            }

    LPCSTR dmsDocId = NULL;

    if (pIodmWrk != NULL)
        {   /* Verify that docId matches theDocId and report. */

            dmsDocId = pIodmWrk -> docID();

            if (strncmp(theDocId, dmsDocId, sizeof(theDocId)) )
                 {  /* Oh Oh, the DocIds don't match.
                       */
                    fputs(  STARS "        " "DIFFERENT DOCID VALUE:\n",
                            stdout
                            );
                    fputs(  docINDENT, stdout);
                    fputs(  dmsDocId, stdout);
                    fputs(  "\n", stdout);
                    }
                    else {  fputs(  docINDENT "docID matches the"
                                           " selected docID.\n",
                                    stdout
                                    );
                            }
            }

    LPCSTR dmsDocLoc = NULL;

    if (pIodmWrk != NULL)
        {   /* Verify that docLocation matches theDocLoc and report. */

            dmsDocLoc = pIodmWrk -> docLocation();

            if (strncmp(theDocLoc, dmsDocLoc, sizeof(theDocLoc)) )
                 {  /* Oh Oh, the DocLocs don't match.
                       */
                    fputs(  STARS "        " "DIFFERENT DOCLOC VALUE:\n",
                            stdout
                            );
                    fputs(  docINDENT, stdout);
                    fputs(  dmsDocLoc, stdout);
                    fputs(  "\n", stdout);
                    }
                    else {  fputs(  docINDENT "docLocation matches"
                                           " the opened docLocation.\n",
                                    stdout
                                    );
                            }
            }


    if (pIodmWrk != NULL)
         {  /* Report all of the Document Properties */

            reportDocProp(stdout, docINDENT,
                          pIodmWrk, "ODM_NAME", ODM_NAME);

            reportDocProp(stdout, docINDENT,
                          pIodmWrk, "ODM_AUTHOR", ODM_AUTHOR);

            reportDocProp(stdout, docINDENT,
                          pIodmWrk, "ODM_TYPE", ODM_TYPE);

            reportDocProp(stdout, docINDENT,
                          pIodmWrk, "ODM_CONTENTFORMAT", ODM_CONTENTFORMAT);

            reportDocProp(stdout, docINDENT,
                          pIodmWrk, "ODM_TITLETEXT", ODM_TITLETEXT);

            }

    if (proceedOK)

         {  /* PLACEHOLDER UNTIL WE HAVE MORE CODE */

            fputs("\n\n" INDENT "> > > > NO FURTHER OPERATION"
                                       " IMPLEMENTED (Check05).\n",
                  stdout );

            }






    /* ACCOUNT FOR THE RELEASE OF RESOURCES THAT ARE BEING HELD. */

    if ( pIodmWrk != NULL )
         {  /* Report the ended-use of the working document.
               We actually release it last so that we confirm the
               proper sequence of releases and the ODMA operations that
               go with them.
               */

               reportGoesIodmWrk(stdout);
               }

    if ( pIodmApp != NULL )
         {  /* Report how the resources will be unwound. */

            if ( pIodmApp -> hasConMan() )

                 {   /* Only cover the legitimate DMS case.  We
                        have complained about any infraction
                        already. */

                     if ( pIodmApp -> hasDefaultDMS() )

                          reportGoesDMS(stdout);

                     reportGoesConMan(stdout);

                     }
            }


    /* GUARDED RELEASE PROCESS.  All interface references are released
       if they are (still) held at the conclusion of the program.

       If any resources have been acquired in the course of
       IodmApplication100 usage, they will be released automatically
       when the last reference to the interface is released.
       */

    fputs("\n", stdout);

    if (pIodmApp != NULL)
        { pIodmApp -> Release(); pIodmApp = NULL;
          reportRelease(stdout, "IodmApplication100");
          ++releaseCount;
          }

    if (pIodmAppUnk != NULL)
        { pIodmAppUnk -> Release(); pIodmAppUnk = NULL;
          reportRelease(stdout, "IUnknown");
          ++releaseCount;
          }

    if (pIFooUnknown != NULL)
        { pIFooUnknown -> Release(); pIFooUnknown = NULL;
          reportRelease(stdout, "IFoo (??)");
          ++releaseCount;
          }

    if (pIodmWrk != NULL)
        { pIodmWrk -> Release(); pIodmWrk = NULL;
          reportRelease(stdout, "IodmWorking100");
          ++releaseCount;
          }

    if (releaseCount > 0)
        fputs("\n", stdout);

    goodbye(stdout);

    return 0;

    } /* main */


/* 0.07 2007-01-02-20:46 Report the document properties.

   0.06 2007-01-02-20:08 Check the docId and docLocation results for
        having the same values used in creating the document object.

   0.05 2007-01-02-18:25 Show the interfaceImplementation and the
        viewOnly status of the IodmWorking100 interface of a successfully-
        established working document.

   0.04 2007-01-02-17:39 Instantiate the IodmWorking100 interface for a
        successfully-opened DMS Document, reporting the result.  Demonstrate
        that the IodmWorking100 interface releases properly and with an
        ODMCloseDoc operation, and that it holds an IodmApplication100i
        interface properly too.

   0.03 2007-01-02-17:17 Report the result of successful OpenDoc.

   0.02 2007-01-02-16:33 Add the OpenDoc case and report the details.
        See what happens with the Odma32.log with an unbalanced open.

   0.01 2007-01-02-14:38 Make initial changes to prepare to open a
        selected document IodmApplication100.selectDoc() has a document
        result: Expand reportIface() to handle different indentations, and
        refactor traceSelectDoc back into Main and confirm that the
        regression case still runs.

   0.00 2006-12-20-16:40 Start by cloning Setup04 0.20 as Check05 and
        producing a clean regression check before we do anything else.

   $Header: /ODMdev/info/odma/OdmNative100/test/check05/Check05.cpp 30    07-01-02 21:39 Orcmid $
   */

/*                        *** end of Check05.cpp ***                  */

