/* odmjni100.cpp 0.08                UTF-8                   dh:2007-01-13
 *
 *
 *                 IMPLEMENT THE NATIVE METHODS OF ODMJNI100
 *                 *****************************************
 *
 * This is the C++ Language implementation of the native-method DLL
 * for odmjni100.  It is developed progressively in line with the
 * expansion of info.odma.odmjni100 OdmJniBind class support for
 * different ODMA operations.
 *
 * We have native methods for several OdmJni classes, including OdmJniBind
 * and OdmJniVew, but we will still have a single DLL.  This maximizes
 * on the statically-bound C++ DLL runtime for the native methods and the
 * OdmNative libraries.  We also reduce the amount of DLL loading.
 */


#include <jni.h>
    /* For the standard JNI interface.  This interface will be the
       C++ version when building odmjni100.dll.
       */

#include "odmjni100x.h"
    /* Rely on the annotated cumulatively-updated header. */

#include <windows.h>
    /* For native interfaces and COM-related definitions. */

#include "OdmNative100.hpp"
    /* For functions and classes of OdmNative and IodmNative. */

#include "OdmWorking100.hpp"
    /* For the IID and the interface pointer, basically */

#include "Odma32types100.h"
    /* For ODMA-specific types and values that we rely on and produce. */



    /* IMPLEMENTATION OF ODMJNIBIND NATIVE METHODS
       ******************************************* */


JNIEXPORT
    jint JNICALL
         Java_info_odma_odmjni100_OdmJniBind_jniVersion
             (  JNIEnv *pIenv,       /* JNI environment interface */
                jclass pThisClass    /* not used */
                )

{   /* Return the JNI Version Number m.n, coded as 0x000m000n.
       pIenv is a ppv. That is, JNIEnv is a pv, and what's passed
       in is a ppv.  In C++, Ienv is a class with interface methods.
       */

    return pIenv->GetVersion();

    } /* jniVersion */



static const IID IID_IodmApplication100 = IID_IodmApplication100_ ;
    /* IodmApplication100 IID of the current header level. */

static const IID IID_IodmWorking100 = IID_IodmWorking100_ ;
    /* IodmWorking100 IID of the current header definition/contract */



                    /* ODMJNIBIND STATIC NATIVE METHODS *
                     * ******************************** *
                     */

JNIEXPORT
    jlong JNICALL
        Java_info_odma_odmjni100_OdmJniBind_getIodmNative
            (   JNIEnv *pIenv,      /* JNI environment interface */
                jclass pThisClass,  /* not used */
               jstring jAppId       /* The Application ID */
               )

{   /* This procedure binds an IodmApplication100 interface using
       the OdmBindNative100 function.
       */

    const char *pszAppId
        = pIenv -> GetStringUTFChars(jAppId, NULL);
            /* Get pointer to the UTF (presumably-ASCII string.
               XXX: We let OdmBindNative100 check the string for
               us.  It should have been checked by OdmJniBind before
               we got to here. */

    IUnknown *pIUnk = NULL;
            /* receptacle for the pIodmNative returned from
               OdmBindNative100.  We ask for IodmApplication100
               here so that other code doesn't have to query
               interface and we have a check on the proper version
               by IID match-up.
               */

    HRESULT rc = OdmBindNative100(  pszAppId,
                                    IID_IodmApplication100,
                                    (void **) &pIUnk
                                    );
            /* If everything is working, we get an interface that is
               reserved for us (and hence our caller). */

    pIenv -> ReleaseStringUTFChars(jAppId, pszAppId);
            /* Let go of the AppID UTF - we don't need it any longer. */

    if (SUCCEEDED(rc)) return (jlong) pIUnk;
            /* Hand out the disguised pIUnk that we succeeded in getting.*/

    return (jlong) ODM_E_FAIL;
            /* Anything else produces a low-numbered value. */

    } /* getIodmNative */



JNIEXPORT
    jlong JNICALL
        Java_info_odma_odmjni100_OdmJniBind_holdNativeInterface
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* A disguised Native pointer */
                )
{   /* This is a standard IUnknown::AddRef() operation.  It could
       also be a QueryInterface as a way of also normalizing the
       interface, but we will go for speed instead. */

    if (rIodmNative == (rIodmNative & 0x1F))
         return 0;
            /* XXX: Simply ignore non-pointer values. */

    ((IUnknown *) rIodmNative) -> AddRef();

    return rIodmNative;

    } /* holdNativeInterface */



JNIEXPORT
    void JNICALL
        Java_info_odma_odmjni100_OdmJniBind_freeNativeInterface
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* a disguised Native pointer */
                )
{   /* Perform a standard IUnknown::Release() using the disguised
       interface reference we are handed.
       */

    if (rIodmNative == (rIodmNative & 0x1F)) return;
            /* XXX: Simply ignore non-pointer values. */

    ((IUnknown *) rIodmNative) -> Release();

    } /* freeNativeInterface */


    /* ODMJNIAPP OPERATIONS THAT COULD BE MADE STATIC THERE *
     * **************************************************** *
     *
     *      FIXME: These static native methods work
     *             better on the OdmJniApp class
     *             where they are actually used.
     *             They can be private there.
     */


JNIEXPORT
    jboolean JNICALL
        Java_info_odma_odmjni100_OdmJniBind_hasConMan
            (   JNIEnv  *pIenv,     /* not used */
                jclass  pThisClass, /* not used */
                 jlong  rIodmNative /* a disguised Native pointer */
                )
{   /* Perform IodmApplication100::hasConMan() and that's all
       we need. */

    /* XXX: Technically, we should do a query interface here, but we
       assert that we have IodmApplication100 every time this
       method is called.
       */

    return ((IodmApplication100 *) rIodmNative) -> hasConMan();


    } /* hasConMan */



JNIEXPORT
    jboolean JNICALL
        Java_info_odma_odmjni100_OdmJniBind_hasDefaultDMS
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* a disguised Native pointer */
                )
{   /* Perform IodmApplication100::hasDefaultDMS() and we're done.
       */

    return ((IodmApplication100 *) rIodmNative) -> hasDefaultDMS();

    } /* hasDefaultDMS */



JNIEXPORT
    jlong JNICALL
        Java_info_odma_odmjni100_OdmJniBind_selectDoc
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* a disguised Native pointer */
                )
{   /* Use IodmApplication100::selectDocID to connect to the DMS and
       return the selection from the user.

       We return either an ODMSTATUS value or we return an
       IodmWorking100 interface reference.

       This implementation is based on the OdmNative100 test Check05 0.07
       approach.  When OdmNative100 is refactored to to use cohesive
       methods, this function will become much simpler.
       */

    if (rIodmNative == (rIodmNative & 0x1F)) return 0;
            /* XXX: Simply ignore non-pointer values. */

    IodmApplication100 *pIodmApp = (IodmApplication100 *) rIodmNative;
            /* Use the interface pointer we are given.
               XXX: We do not QueryInterface because we trust the caller
                    to provide the correct goodies.  This function should
                    really be a native method of the OdmJniApp class.
               */

    char theDocId[ODM_DOCID_MAX+2]={'\0'};
            /* FIXME: We need serious buffer over-run
                      protection here, although when we refactor this will
                      all be handled by the IodmAppication100
                      implementation..
               */

    BOOL viewMode = TRUE;
            /* FIXME: We need a better approach to this when we allow
               other cases.  It may be trickier to accomplish this
               when the DMS may push back on what we want.
               */

    ODMSTATUS rc = pIodmApp -> selectDocID(theDocId, &viewMode);

    if (rc != ODM_SUCCESS)
         switch (rc)
         {  /* Stop now with the failure case we have */

            case      ODM_E_FAIL:
            case    ODM_E_CANCEL:
            case ODM_E_APPSELECT:
                        return (jlong) rc;
            default:    return (jlong) ODM_E_FAIL;
            }

    /* We have a good Document ID.  Now we need to see if we can
       open the document.
       */

    char theDocLoc[ODM_FILENAME_MAX+20] = {'\0'};

    rc = pIodmApp -> openDoc(theDocId, theDocLoc, viewMode);
            /* The select succeeded, so the open should work */

    if (rc != ODM_SUCCESS)
         return (jlong) ODM_E_FAIL;

    IodmWorking100 *pIodmWrk = NULL;

    HRESULT hr = pIodmApp -> makeWorkingDocument
                                (  theDocId, theDocLoc, viewMode,
                                   IID_IodmWorking100,
                                   (void**) &pIodmWrk
                                   );

    if (SUCCEEDED(hr))
         return (jlong) pIodmWrk;
    else return (jlong) ODM_E_FAIL;


    } /* selectDoc */


                    /* ODMJNIAPP STATIC NATIVE METHODS *
                     * ******************************* *
                     *
                     * This method is private to OdmJniApp
                     * and others should be here too.
                     */


JNIEXPORT
    jboolean JNICALL
        Java_info_odma_odmjni100_OdmJniApp_checkViewOnly
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* An IodmWorking100 reference */
                )

{   /* It's available directly in the IodmWorking100 interface.
       */

    return ((IodmWorking100 *) rIodmNative) -> viewOnly();

    } /* checkViewOnly */



                    /* ODMJNIVIEW STATIC NATIVE METHODS *
                     * ******************************** *
                     *
                     * XXX: These methods are private to
                     *      OdmJniView, the only class
                     *      that relies on them.
                     */

JNIEXPORT
    jstring JNICALL
        Java_info_odma_odmjni100_OdmJniView_jniDocID
            (   JNIEnv *pIenv,      /* Used to make string */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* an IodmWorking100 reference */
                )

{   /* Take the IodmWorking100::docID() right through to conversion
       to Java Unicode.
       */

    return pIenv -> NewStringUTF
                        (  ((IodmWorking100 *) rIodmNative)
                                -> docID()
                           );

    /* XXX: This approach depends on the ODMA Document ID being limited
            to ISO 646/ASCII Character codes that are one-to-one
            equivalent to Unicode code points.
       */

    } /* jniDocID */




JNIEXPORT
    jstring JNICALL
        Java_info_odma_odmjni100_OdmJniView_jniDocLoc
            (   JNIEnv *pIenv,      /* Used to make string */
                jclass pThisClass,  /* Not used */
                 jlong rIodmNative  /* an IodmWorking100 reference */
                 )

{   /* Take the IodmWorking100::docLocation() right through to conversion
       to Java Unicode.
       */

    return pIenv -> NewStringUTF
                        (  ((IodmWorking100 *) rIodmNative)
                                -> docLocation()
                           );

    /* XXX: This approach depends on the ODMA Document Location being
            limited to ISO 646/ASCII Character codes that are one-to-one
            equivalent to Unicode code points.
       */

    } /* jniDocLoc */


JNIEXPORT
    jstring JNICALL
        Java_info_odma_odmjni100_OdmJniView_jniDocProp
            (   JNIEnv *pIenv,      /* Used to make string */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative, /* an IodmWorking100 reference */
                  jint propNum      /* item number of desired property */
                )

{   /* Use OdmWorking100::docProperty to obtain the item, if available. */

    char thePropVal[300] = {'\0'};
            /* Provide generous storage for the string */

    ODMSTATUS rc = ((IodmWorking100 *) rIodmNative)
                    -> docProperty(propNum, thePropVal, sizeof(thePropVal));

    if (rc != ODM_SUCCESS) return (jstring) NULL;

    return pIenv -> NewStringUTF(thePropVal);

    } /* jniDocProp */


                    /* ODMJNIWOKRK STATIC NATIVE METHODS *
                     * ******************************** *
                     *
                     * XXX: This method is private to
                     *      OdmJniWork, the only class
                     *      that relies on it.
                     */


JNIEXPORT
    jboolean JNICALL
        Java_info_odma_odmjni100_OdmJniWork_jniCommitChanges
            (   JNIEnv *pIenv,      /* not used */
                jclass pThisClass,  /* not used */
                 jlong rIodmNative  /* for an IodmWorking100 */
                )

{   /* It's available directly in the IodmWorking100 interface.
       */

    return ((IodmWorking100 *) rIodmNative) -> commitChanges();

    } /* commitChanges */



/* 0.08 2007-01-13-18:27 addition of checkViewOnly() and commitChanges()
        for completion of all methods needed for the 0.40alpha release.
   0.07 2007-01-09-20:52 Introduce the three native methods used by
        OdmJniView 0.04 for the 0.30alpha release.  The DLL is still 68k.
   0.06 2007-01-08-21:44 Update with complete selectDoc implementation.  The
        solutions is a variant on that used in OdmNative100 Check05 0.07.
   0.05 2006-11-30-00:42 Add selectDoc implementation.  This is provisional
        for now because we don't have any OdmNative document objects to
        instantiate yet.  This DLL is 68k.  We don't want multiple DLLs so
        we will have this one be enough.
   0.04 2006-11-29-21:56 Add hadDefaultDMS implementation.
   0.03 2006-11-29-20:58 Add hasConMan implementation.
   0.02 2006-11-29-19:00 Add holdNativeInterface implementation.
   0.01 2006-11-29-13:10 Add getIodmNative and freeNativeInterface.  These
        provide enough to instantiate an OdmConnection implementation.  Now
        there are 69,632 bytes in the DLL.
   0.00 2006-11-28-17:04 Strip down from odmjni100 0.06 to include only
        the jniVersion() method for initial confirmation of DLL location.
        The DLL compiles to the magical 49,152 bytes.

   $Header: /ODMdev/info/odma/odmjni100/odmjni100.cpp 10    07-01-13 18:44 Orcmid $
   */

/*                         *** end of odmjni100.cpp ***                  */
