/* wfsapp.cpp - Implementation of the application object for the
 *  sample WMS.
 *
 * Written by D. Hamson, Novell
 *
 * COPYRIGHT (C) 1995
 * AIIM International
 * All Rights Reserved
 */


#include <windows.h>
#include "odmwfcom.h"
#include "wfserver.h"
#include "wfserver.rh"

extern HINSTANCE NEAR hInst;
// Global to track existing workflows
WorkflowList *WorkList;

// main constructor
Application::Application(LPUNKNOWN pUnkOuter, DWORD dwEnvData)
: m_Unknown(this), m_ODMWFWorkMan(this)
{
	m_dwRefs = 1;

	if(pUnkOuter == NULL) {
		m_pUnkOuter = &m_Unknown;
	} else {
		m_pUnkOuter = pUnkOuter;
	}

	// capture calling application's window handle
	m_clientWind = (HWND) dwEnvData;

	// create the existing workflow list
	WorkList = new WorkflowList;

}

// destructor
Application::~Application(void)
{
	if (WorkList)
		delete WorkList;
}

HRESULT Application::GetInterface(REFIID riid, LPVOID FAR *ppvObj)
{
	return m_Unknown.QueryInterface(riid,ppvObj);
}

/*  IUnknown Interface implementation */

STDMETHODIMP NC(Application,CUnknown)::QueryInterface(REFIID riid,
LPVOID FAR* ppobj)
{
	HRESULT hRes = NOERROR;

	*ppobj = NULL;

	if(riid == IID_IUnknown) {
		*ppobj = (LPVOID)&m_pObject->m_Unknown;
	} else if(riid == IID_IODMWFWorkMan) {
		*ppobj = (LPVOID)&m_pObject->m_ODMWFWorkMan;
	}

	if (*ppobj == NULL) {
		hRes = ResultFromScode(E_NOINTERFACE);
	} else {
		((LPUNKNOWN)*ppobj)->AddRef();
	}

	return hRes;
}

STDMETHODIMP_(ULONG) NC(Application,CUnknown)::AddRef(VOID)
{
	ULONG dwRefs = ++m_pObject->m_dwRefs;

	return dwRefs;
}

STDMETHODIMP_(ULONG) NC(Application,CUnknown)::Release(VOID)
{
	ULONG dwRefs = --m_pObject->m_dwRefs;

	if (dwRefs == 0)
		delete m_pObject;

	return dwRefs;
}

/* IODMWFWorkMan Interface Implementation */

STDMETHODIMP NC(Application, CODMWFWorkMan)::QueryInterface(REFIID riid,
LPVOID FAR* ppobj)
{
	return m_pObject->m_pUnkOuter->QueryInterface(riid,ppobj);
}

STDMETHODIMP_(ULONG) NC(Application, CODMWFWorkMan)::AddRef(VOID)
{
	return m_pObject->m_pUnkOuter->AddRef();
}

STDMETHODIMP_(ULONG) NC(Application, CODMWFWorkMan)::Release(VOID)
{
	return m_pObject->m_pUnkOuter->Release();
}

// create a new workflow id and return it.
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::NewObject(LPSTR lpszWorkflowId,
LPDWORD pdwFlags)
{
	Workflow *wf;
	ODMSTATUS err = ODM_E_FAIL;

	wf = WorkList->NewWorkflow();

	if (wf) {
		strcpy(lpszWorkflowId,wf->GetWfId());
			MessageBox(m_pObject->m_clientWind,lpszWorkflowId,"New Workflow",
				MB_APPLMODAL+MB_OK);
		err = 0;
	}

	return err;
}

// display the list of workflows and allow one to be selected
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::SelectObject(LPSTR lpszWorkflowId,
LPDWORD pdwFlags)
{
	ODMSTATUS err;

	if (*pdwFlags & ODM_SILENT)
		err = ODM_E_USERINT;
	else {
		err = DialogBoxParam(hInst, MAKEINTRESOURCE(SELECT_DIALOG),
					m_pObject->m_clientWind, (DLGPROC)SelectWorkflowProc,
					(LPARAM)lpszWorkflowId);

		if (err == IDOK)
			err = 0;
		else
			err = ODM_E_CANCEL;
	}

	return err;
}

// Add a document to a workflow
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::InsertItem(LPSTR lpszWorkflowId,
WORD wItemNumber, LPSTR lpszDocIdentifier)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->InsertDocument(wItemNumber,lpszDocIdentifier);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Remove a document from a workflow
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::RemoveItem(LPSTR lpszWorkflowId,
WORD wItemNumber)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->RemoveDocument(wItemNumber);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Get a document assigned to a workflow
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::GetItem(LPSTR lpszWorkflowId,
WORD wItemNumber, LPSTR lpszDocIdentifier)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->GetDocument(wItemNumber,lpszDocIdentifier);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Process workflow, causes the workflow to go to the next step.
// In this WMS all flows are the same and have only three steps.
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::ProcessObject(LPSTR lpszWorkflowId,
LPDWORD pdwFlags)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->ChangeStatus();
		if (!err)
			MessageBox(m_pObject->m_clientWind,wf->GetStatus(),"Process Workflow",
					MB_APPLMODAL+MB_OK);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Get the property value for the specified property
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::GetObjectInfo(LPSTR lpszWorkflowId,
WORD attribute, LPSTR lpszData, WORD datalen)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->GetInfo(attribute,lpszData,datalen);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Set the value of the specified property
STDMETHODIMP_(ODMSTATUS) NC(Application, CODMWFWorkMan)::SetObjectInfo(LPSTR lpszWorkflowId,
WORD attribute, LPSTR lpszData)
{
	Workflow *wf;
	ODMSTATUS err = 0;

	wf = WorkList->FindWorkflow(lpszWorkflowId);

	if (wf) {
		err = wf->SetInfo(attribute,lpszData);
	} else {
		err = ODM_E_WORKFLOWID;
	}

	return err;
}

// Implementation of the activate call.
STDMETHODIMP_(void) NC(Application, CODMWFWorkMan)::Activate(WORD action,
LPSTR lpszWorkflowId)
{

	switch(action) {
	 	case ODM_NONE:
			MessageBox(m_pObject->m_clientWind,"WMS Active: ODM_NONE","Activate",
					MB_APPLMODAL+MB_OK);
			break;
		case ODM_DELETE:  //  delete workflow
			WorkList->RemoveWorkflow(lpszWorkflowId);
			break;
		case ODM_SHOWATTRIBUTES:  // 
			MessageBox(m_pObject->m_clientWind,"WMS Active: ODM_SHOWATTRIBUTES","Activate",
					MB_APPLMODAL+MB_OK);
			break;
		case ODM_EDITATTRIBUTES:
			MessageBox(m_pObject->m_clientWind,"WMS Active: ODM_EDITATTRIBUTES","Activate",
					MB_APPLMODAL+MB_OK);
			break;
	}
}

// report information about this WMS
STDMETHODIMP_(void) NC(Application, CODMWFWorkMan)::GetWMSInfo(LPSTR wmsId,
LPWORD pwVerNo, LPDWORD pdwExtensions)
{
	strcpy(wmsId, WMSID);
	*pwVerNo = ODMAWF_API_VERSION;
	*pdwExtensions = 0;
}


