/* wfswf.cpp - Implementation of the Workflow Object for the sample WMS.
 *	The Id of a Workflow is generated based on time.  These objects are
 *  persistent and use an .INI file as a data store.
 *
 * Written by D. Hamson
 *
 * COPYRIGHT (C) 1995
 * AIIM International
 * All Rights Reserved
 */


#include <windows.h>
#include <dos.h>
#include "wfserver.h"


// constructor for new workflow object
Workflow::Workflow(void)
{
	int i;
#ifdef WIN32
	SYSTEMTIME tm;
#elif __BORLANDC__
	struct time tm;
#else
	struct _dostime_t tm;
#endif


#ifdef WIN32
	GetLocalTime(&tm);

	if (tm.wMilliseconds > 99)
		tm.wMilliseconds /= 10;

	wsprintf(WfId,"::ODMAWF\\%s\\%02d-%02d-%02d-%02d",WMSID,tm.wHour,tm.wMinute,tm.wSecond,tm.wMilliseconds);
#elif __BORLANDC__
	gettime(&tm);
	wsprintf(WfId,"::ODMAWF\\%s\\%02d-%02d-%02d-%02d",WMSID,tm.ti_hour,tm.ti_min,tm.ti_sec,tm.ti_hund);
#else
	_dos_gettime(&tm);
	wsprintf(WfId,"::ODMAWF\\%s\\%02d-%02d-%02d-%02d",WMSID,tm.hour,tm.minute,tm.second,tm.hsecond);
#endif

	/* provide some defaults for a new workflow object */
	wsprintf(Name,"Name - %s",WfId);
	wsprintf(TitleText,"Title Text - %s",WfId);
	strcpy(Status,"New");
	strcpy(Author,"Author Name");
	strcpy(UserId,"USERID");
	strcpy(UserName,"User Name");

	ItemCount = 0;

	for(i=0;i<MAXDOCS;i++)
		List[i] = NULL;

	return;
}

// constructor for a workflow object that exists in the data store (ini file)
Workflow::Workflow(LPSTR lpszWorkfId)
{
	int i;
	char nbuf[8], *docbuf;

	strcpy(WfId, lpszWorkfId);

	GetPrivateProfileString(WfId,"Author","",Author,sizeof(Author),"WFSERVER.INI");
	GetPrivateProfileString(WfId,"Name","",Name,sizeof(Name),"WFSERVER.INI");
	GetPrivateProfileString(WfId,"Status","",Status,sizeof(Status),"WFSERVER.INI");
	GetPrivateProfileString(WfId,"TitleText","",TitleText,sizeof(TitleText),"WFSERVER.INI");
	GetPrivateProfileString(WfId,"UserId","",UserId,sizeof(UserId),"WFSERVER.INI");
	GetPrivateProfileString(WfId,"UserName","",UserName,sizeof(UserName),"WFSERVER.INI");

	ItemCount = 0;
	docbuf = new char[MAXDOCID];
	for(i=0;i<MAXDOCS;i++) {
		List[i] = NULL;
		wsprintf(nbuf,"DOC%d",i);
		GetPrivateProfileString(WfId,nbuf,"",docbuf,MAXDOCID,"WFSERVER.INI");
		if (*docbuf) {
			List[i] = new Document(docbuf);
			ItemCount++;
		}
	}
	delete [] docbuf;
}

// put workflow information into .ini file
void Workflow::WriteInfo(void)
{
	int i;
	char nbuf[8];

	WritePrivateProfileString("WorkflowList",WfId,"","WFSERVER.INI");

	WritePrivateProfileString(WfId,"Author",Author,"WFSERVER.INI");
	WritePrivateProfileString(WfId,"Name",Name,"WFSERVER.INI");
	WritePrivateProfileString(WfId,"Status",Status,"WFSERVER.INI");
	WritePrivateProfileString(WfId,"TitleText",TitleText,"WFSERVER.INI");
	WritePrivateProfileString(WfId,"UserId",UserId,"WFSERVER.INI");
	WritePrivateProfileString(WfId,"UserName",UserName,"WFSERVER.INI");

	for(i=0;i<MAXDOCS;i++) {
		if (List[i]) {
			wsprintf(nbuf,"DOC%d",i);
			WritePrivateProfileString(WfId,nbuf,List[i]->GetDocId(),"WFSERVER.INI");
		}
	}

}

// add a document to the list associated with a workflow
ODMSTATUS Workflow::InsertDocument(WORD wItemNumber,LPSTR lpszDocIdentifier)
{
	WORD i;
	ODMSTATUS err = 0;

		// MAXDOCS is defined by this WMS, not by ODMAWF
		if ((wItemNumber > ItemCount) || (ItemCount == MAXDOCS)) {
			err = ODM_E_BADINDEX;
		} else {
			for(i=MAXDOCS-1;i>wItemNumber;i--) {
				List[i] = List[i-1];
			}
			List[wItemNumber] = new Document(lpszDocIdentifier);
         ItemCount++;
		}

	return err;
}

// remove a document from this workflow
ODMSTATUS Workflow::RemoveDocument(WORD wItemNumber)
{
	int i;
	ODMSTATUS err = 0;

		if ((wItemNumber > ItemCount) || (ItemCount == 0)) {
			err = ODM_E_BADINDEX;
		} else {
			delete List[wItemNumber];
			ItemCount--;
			for(i=wItemNumber;i<MAXDOCS-1;i++) {
				List[i] = List[i+1];
			}
			List[MAXDOCS-1] = NULL;
		}

	return err;
}

// return the document id of a document for this workflow
ODMSTATUS Workflow::GetDocument(WORD wItemNumber,LPSTR lpszDocIdentifier)
{
	ODMSTATUS err = 0;
	char *DocId;

		if ((wItemNumber > (ItemCount-1)) || (ItemCount == 0)) {
			err = ODM_E_BADINDEX;
		} else {
			DocId = List[wItemNumber]->GetDocId();
			if (DocId)
				strcpy(lpszDocIdentifier,DocId);
		}

	return err;
}

// a real WMS would do something more substatial, like forward this to the next
// user, but this system just changes the status value to show the call was made.
// This WMS has only three states: New (process has not be called yet), Active (
// process has been called once) and Complete (process has been called twice or more.)
ODMSTATUS Workflow::ChangeStatus(void)
{
	if (!strcmp(Status,"New"))
		strcpy(Status,"Active");
	else
		if (!strcmp(Status,"Active"))
			strcpy(Status,"Complete");

	return 0;
}

// return requested property value
ODMSTATUS Workflow::GetInfo(WORD attribute,LPSTR lpszData,WORD datalen)
{
	ODMSTATUS err = 0;

	if (lpszData && datalen) {
		memset(lpszData,0,datalen);
		switch(attribute) {
			case ODM_AUTHOR:
				strncpy(lpszData,Author,datalen-1);
				break;
			case ODM_NAME:
				strncpy(lpszData,Name,datalen-1);
				break;
			case ODM_STATUS:
				strncpy(lpszData,Status,datalen-1);
				break;
			case ODM_TITLETEXT:
				strncpy(lpszData,TitleText,datalen-1);
				break;
			case ODM_ITEMCOUNT:
				if (datalen<2)
					err = ODM_E_FAIL;
				else
					wsprintf(lpszData,"%d",ItemCount);
				break;
			case ODM_USERID:
				strncpy(lpszData,UserId,datalen-1);
				break;
			case ODM_USERNAME:
				strncpy(lpszData,UserName,datalen-1);
				break;
			default:
				err = ODM_E_ITEM;
				break;
		}
	}
	return err;
}

// set requested property value
ODMSTATUS Workflow::SetInfo(WORD attribute,LPSTR lpszData)
{
	ODMSTATUS err = 0;

	if (lpszData) {
		switch(attribute) {
			case ODM_AUTHOR:
				strncpy(Author,lpszData,sizeof(Author)-1);
				break;
			case ODM_NAME:
				strncpy(Name,lpszData,sizeof(Name)-1);
				break;
			default:
				err = ODM_E_ITEM;
				break;
		}
	}
	return err;
}




